#!/usr/bin/env escript
%% -*- erlang -*-

%% %CopyrightBegin%
%%
%% SPDX-License-Identifier: Apache-2.0
%%
%% Copyright Ericsson AB 2024-2025. All Rights Reserved.
%%
%% Licensed under the Apache License, Version 2.0 (the "License");
%% you may not use this file except in compliance with the License.
%% You may obtain a copy of the License at
%%
%%     http://www.apache.org/licenses/LICENSE-2.0
%%
%% Unless required by applicable law or agreed to in writing, software
%% distributed under the License is distributed on an "AS IS" BASIS,
%% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%% See the License for the specific language governing permissions and
%% limitations under the License.
%%
%% %CopyrightEnd%

%% Tool to generate fixes to the OTP SPDX file produced by ORT,
%% and tests to verify that given the scan-results generated by
%% ORT and the OTP SPDX, the OTP SPDX is compliant with SPDX 2.2
%%
%% Notice that ORT produces correct SPDX 2.2, this Erlang/OTP SPDX
%% needs to be futher split into packages. this need comes from users
%% wanting to opt-out of some applications, and also to keep track
%% of which packages are vendor packages/files.
%%
%% Because of this need, it is easy to break the correct SPDX
%% simply by, e.g., adding a package name that contains underscores.
%% To prevent from these issues, we have a test option that checks
%% (to some degree) that the SPDX generate is correct.
%%
%% After this validation, users of this script should still run
%% other validator tools, such as, ntia-conformance-checker.
%%

%%
%% REUSE-IgnoreStart
%%
%% Ignore copyright detection heuristics of REUSE tool in this file.
%% this is needed to avoid REUSE false positives on 'Copyright' variable name.
%%

-include_lib("kernel/include/file.hrl").

-export([test_project_name/1,
         test_name/1,
         test_creators_tooling/1,
         test_spdx_version/1]).

-export([test_minimum_apps/1, test_copyright_not_empty/1, test_filesAnalised/1,
         test_hasFiles_not_empty/1, test_homepage/1,
         test_licenseConcluded_exists/1, test_licenseDeclared_exists/1,
         test_licenseInfoFromFiles_not_empty/1, test_package_names/1,
         test_package_ids/1, test_verificationCode/1, test_supplier_Ericsson/1,
         test_originator_Ericsson/1, test_versionInfo_not_empty/1, test_package_hasFiles/1,
         test_project_purl/1, test_packages_purl/1, test_download_location/1, 
         test_package_relations/1, test_has_extracted_licenses/1,
         test_vendor_packages/1, test_erts/1, test_download_vendor_location/1
         %% test_copyright_format/1, test_files_licenses/1,
        ]).

%% openvex tests
-export([test_openvex_branched_otp_tree/0,
         test_openvex_branched_otp_tree_idempotent/0]).

%%
%% SBOM SPDX MACROS
%%
-define(default_classified_result, "scan-result-classified.json").
-define(default_scan_result, "scan-result.json").
-define(diff_classified_result, "scan-result-diff.json").
-define(erlang_license, ~"Apache-2.0").
-define(spdxref_project_name, ~"SPDXRef-Project-OTP").
-define(spdx_project_name, ~"Erlang/OTP").
-define(spdx_creators_tooling, ~"Tool: otp_compliance").
-define(spdx_supplier, ~"Organization: Ericsson AB").
-define(spdx_download_location, ~"https://github.com/erlang/otp/releases").
-define(spdx_homepage, ~"https://www.erlang.org").
-define(spdx_purl_meta_data, ~"?vcs_url=git+https://github.com/erlang/otp.git").
-define(spdx_version, ~"SPDX-2.3").
-define(otp_version, 'OTP_VERSION'). % file name of the OTP version
-define(spdx_project_purl, #{ ~"comment" => ~"",
                              ~"referenceCategory" => ~"PACKAGE-MANAGER",
                              ~"referenceLocator" => ~"pkg:github/erlang/otp",
                              ~"referenceType" => ~"purl"}).
%%
%%

%%
%% VEX MACROS
%%
-define(VexPath, ~"vex/").
-define(OpenVEXTablePath, "make/openvex.table").
-define(ErlangPURL, "pkg:github/erlang/otp").

-define(FOUND_VENDOR_VULNERABILITY_TITLE, "Vendor vulnerability found").
-define(FOUND_VENDOR_VULNERABILITY, lists:append(string:replace(?FOUND_VENDOR_VULNERABILITY_TITLE, " ", "+", all))).

-define(OTP_GH_URI, "https://raw.githubusercontent.com/" ++ ?GH_ACCOUNT ++ "/refs/heads/master/").

%% GH default options
-define(GH_ADVISORIES_OPTIONS, "state=published&direction=desc&per_page=100&sort=updated").

%% Advisories to download from last X years.
-define(GH_ADVISORIES_FROM_LAST_X_YEARS, 5).

%% Defines path of script to create PRs for missing openvex/vulnerabilities
-define(CREATE_OPENVEX_PR_SCRIPT_FILE, ".github/scripts/create-openvex-pr.sh").

%% Sets end point account to fetch information from GH
%% used by `gh` command-line tool.
%% change to your fork for testing, e.g., `kikofernandez/otp`
-define(GH_ACCOUNT, "erlang/otp").
%%
%%

%% Add more relations if necessary.
-type spdx_relations() :: #{ 'DOCUMENTATION_OF' => [],
                             'CONTAINS' => [],
                             'TEST_OF' => [],
                             'PACKAGE_OF' => []}.

-record(spdx_package, {'SPDXID'           :: unicode:chardata(),
                       'versionInfo'      :: unicode:chardata(),
                       'description'      :: unicode:chardata(),
                       'name'             :: unicode:chardata(),
                       'copyrightText'    :: unicode:chardata(),
                       'filesAnalyzed'    = false :: boolean(),
                       'hasFiles'         = [] :: [unicode:chardata()],
                       'purl'             = false :: false | unicode:chardata(),
                       'homepage'         :: unicode:chardata(),
                       'licenseConcluded' :: unicode:chardata(),
                       'licenseDeclared'  :: unicode:chardata(),
                       'licenseInfoFromFiles' = [] :: [unicode:chardata()],
                       'downloadLocation' = ~"https://github.com/erlang/otp/releases" :: unicode:chardata(),
                       'packageVerificationCode' :: #{ 'packageVerificationCodeValue' => unicode:chardata()},
                       'supplier' = ~"Organization: Ericsson AB" :: unicode:chardata(),
                       'relationships' = #{ 'DOCUMENTATION_OF' => [],
                                            'CONTAINS' => [],
                                            'TEST_OF' => [],
                                            'PACKAGE_OF' => []} :: spdx_relations()
                      }).
-type spdx_package() :: #spdx_package{}.

-record(app_info, { description  :: unicode:chardata(),
                    id           :: unicode:chardata(),
                    vsn          :: unicode:chardata(),

                    %% modules can only be included in one app.
                    %% not_loaded indicates a special handling of this module, e.g., erts.
                    modules      :: [atom()] | not_loaded,
                    applications :: [atom()],
                    included_applications :: [atom()],
                    optional_applications :: [atom()] }).

-type app_info() :: #app_info{}.

-type cve() :: #{ 'CVE' => binary(),
                  'appName' => binary(),
                  'affectedVersions' => [binary()],
                  'fixedVersions' => [binary()]}.

%%
%% Commands
%%
%% sbom
%%
%%    otp-info: given an oss-review-toolkit (ORT) scan result and a
%%              source SBOM, it populates the fields that ORT can't
%%              in Unmanaged projects.
%%
%% compliance   useful for CI/CD compliance checks.
%%
%%    detect:   given a scan-result from ORT, it detects files without license
%%              and writes them into disk.
%%
%%    check:    given a recent scan-result from ORT (possibly from PR), and an
%%              existing file with known files without licenses (from prev. commit),
%%              calculate if new files without licenses have been added to the repo.
%%
%% explore
%%
%%    classify: takes as input a scan of ort and returns a json file containing
%%              as keys the licenses and as values the files under those licenses.
%%
%%    diff:     performs a diff of existing classification file against
%%              other classification files. this is useful to guarantee that
%%              files that had license X had not unexpectedly been reported differently.
%%

%%
%% USE OF COMMANDS
%%
%% The commands `classify` and `diff` are useful for exploring the licenses.
%% ORT does not report in an easy way which files have been attached to which licenses,
%% unless one generates a report. At the time, we cannot generate an SBOM,
%% so we are in the dark.
%%
%% The commands `detect` and `check` can be used in CI/CD to
%% prevent entering new files with unknown license. In the normal case,
%% the `detect` command only needs to be issued once in the repo.
%% Once we keep track of this file, the command is not needed anymore,
%% as the list of files with no license should not grow, and only
%% the `check` command should be executed in the CI/CD.
%%
%%

main(Args) ->
    argparse:run(Args, cli(), #{progname => otp_compliance}).

cli() ->
    #{ commands =>
           #{"sbom" =>
                 #{ help => """
                            Contains useful commands to fix an ORT generated source SBOM.

                            """,
                   commands =>
                        #{"otp-info" =>
                              #{ help =>
                                     """
                                     Adds information missing in ORT's Erlang/OTP source SBOM
                                       - Add homepage
                                       - Fixes license of `*.beam` files
                                       - Fixes project name

                                     Example:

                                     > .github/scripts/otp-compliance.es sbom otp-info --sbom-file bom.spdx.json --input-file scan-result.json
                                     """,
                                 arguments => [ sbom_option(),
                                                write_to_file_option(),
                                                input_option() ],
                                 handler => fun sbom_otp/1},

                          "test-file" =>
                              #{ help =>
                                     """
                                     Verify that the produced SBOM satisfies some minimum requirements

                                     Example:

                                     > .github/scripts/otp-compliance.es sbom test-file --sbom-file otp.spdx.json
                                     """,
                                 arguments => [ sbom_option(), ntia_checker() ],
                                 handler => fun test_file/1},

                          "vendor" =>
                              #{ help =>
                                     """
                                     SBoM contains only vendor dependencies

                                     Example:

                                     > .github/scripts/otp-compliance.es sbom vendor --sbom-file otp.spdx.json
                                     """,
                                 arguments => [ sbom_option()],
                                 handler => fun sbom_vendor/1},

                          "osv-scan" =>
                              #{ help =>
                                     """
                                     Performs vulnerability scanning on vendor libraries.
                                     As a side effect,

                                     Example:

                                     > .github/scripts/otp-compliance.es sbom osv-scan --version maint-28
                                     """,
                                 arguments => [ versions_file(), fail_option() ],
                                 handler => fun osv_scan/1}
                         }},
             "vex" =>
                 #{
                   help => """
                           Create VEX statements
                           Update CVEs and generate OpenVex Statements
                           """,
                   commands =>
                       #{"init" =>
                             #{ help =>
                                    """
                                    Initialise an openvex file.
                                    """,
                                arguments => [ input_option(~"make/openvex.table"), branch_option(), vex_path_option()],
                                handler => fun init_openvex/1},
                         "run" =>
                             #{ help =>
                                    """
                                    Updates an openvex file.
                                    """,
                                arguments => [ input_option(~"make/openvex.table"), branch_option(), vex_path_option()],
                                handler => fun run_openvex/1},

                         "verify" =>
                             #{ help =>
                                    """
                                    Download Github Advisories for erlang/otp.
                                    Download OpenVEX statement from erlang/otp for the selected branch.
                                    Checks that those Advisories are present in OpenVEX statements.
                                    Creates PR for any non-present Github Advisory.

                                    Example:
                                    > .github/scripts/otp-compliance.es vex verify -p

                                    """,
                                arguments => [create_pr()],
                                handler => fun verify_openvex/1
                              },

                         "test" =>
                             #{handler => fun test_openvex/1}
                        }
                  },
             "explore" =>
                 #{  help => """
                            Explore license data.
                            Useful to figure out the mapping files-to-licenses.

                            """,
                    commands =>
                        #{"classify-license" =>
                              #{ help =>
                                     """
                                     Classify files by their license group.
                                       - Input file expects a scan-result from ORT.
                                       - Output file shows mapping between licenses and files.
                                         The output file can be fed to the `explore diff` command.

                                     """,
                                 arguments => [ input_option(?default_scan_result),
                                                output_option(?default_classified_result),
                                                apply_excludes(),
                                                apply_curations() ],
                                 handler => fun classify_license/1},
                          "classify-license-copyright" =>
                              #{ help =>
                                     """
                                     Pair files with their copyright and license.
                                     Depends on a `scan-result.json` and the output of the `classify-license`.

                                     """,
                                 arguments => [ input_option(?default_scan_result),
                                                base_file(?default_classified_result),
                                                output_option() ],
                                 handler => fun classify_path_license_copyright/1},

                          "diff" =>
                              #{ help =>
                                     """
                                     Compare against previous license results.
                                       - Input file should be the output of the `classify` command for input and base files.
                                       - Output returns a summary of additions and deletions per license.

                                     """,
                                 arguments => [ input_option(?default_classified_result),
                                                base_file(),
                                                output_option(?diff_classified_result) ],
                                 handler => fun diff/1}
                         }
                  },
             "compliance" =>
                 #{ help => """
                            Commands to enforce compliance policy towards unlicensed files.

                            """,
                    commands =>
                        #{"detect" =>
                              #{ help =>
                                     """
                                     Detects unlicensed files.
                                     - Input file expects a scan-result from ORT.
                                     - Output file is a list of files without license.
                                       The output file can be fed to the `compliance check` command.

                                     """,
                                 arguments => [ input_option(?default_scan_result),
                                                output_option(),
                                                apply_excludes() ],
                                 handler => fun detect_no_license/1},
                          "check" =>
                              #{ help =>
                                     """
                                     Checks that no new unlicensed files have been added.
                                     - Input file expects scan-result from ORT.
                                     - Base file expects output file from `no_license` command.

                                     """,
                                 arguments => [ input_option(?default_scan_result),
                                                base_file(),
                                                apply_excludes(),
                                                output_option() ],
                                 handler => fun check_no_license/1}}}}}.

%%
%% Options
%%
input_option() ->
    #{name => input_file,
      type => binary,
      long => "-input-file"}.


input_option(Default) ->
    (input_option())#{default => Default}.

sbom_option() ->
    #{name => sbom_file,
      type => binary,
      default => "bom.spdx.json",
      long => "-sbom-file"}.

versions_file() ->
    #{name => version,
      type => binary,
      long => "-version"}.

fail_option() ->
    #{name => fail_if_cve,
      type => boolean,
      default => false,
      long => "-fail_if_cve"}.
%% useful for pull requests since we do not want to
%% add Github Security per found CVE on each PR.

ntia_checker() ->
    #{name => ntia_checker,
      type => boolean,
      default => true,
      long => "-ntia-checker"}.

write_to_file_option() ->
    #{name => write_to_file,
      type => binary,
      default => true,
      long => "-write_to_file"}.

output_option(Default) ->
    #{name => output_file,
      type => binary,
      default => Default,
      long => "-output-file"}.

output_option() ->
    #{name => output_file,
      type => binary,
      required => true,
      long => "-output-file"}.

apply_excludes() ->
    #{name => exclude,
      type => boolean,
      short => $e,
      default => true,
      long => "-apply-excludes"}.

apply_curations() ->
    #{name => curations,
      type => boolean,
      short => $c,
      default => true,
      long => "-apply-curations"}.

base_file() ->
    #{name => base_file,
      type => binary,
      long => "-base-file"}.
base_file(DefaultFile) ->
    #{name => base_file,
      type => binary,
      default => DefaultFile,
      long => "-base-file"}.

branch_option() ->
    #{name => branch,
      type => binary,
      required => true,
      short => $b,
      long => "-branch"}.

vex_path_option() ->
    #{name => vex_path,
      type => binary,
      required => false,
      default => ?VexPath,
      help => "Path to folder containing openvex statements, e.g., `vex/`",
      long => "-vex-path"}.

create_pr() ->
    #{name => create_pr,
      short => $p,
      type => boolean,
      default => false,
      help => "Indicates if missing OpenVEX statements create and submit a PR"}.

%%
%% Commands
%%

sbom_vendor(#{sbom_file  := SbomFile}) ->
    Sbom = decode(SbomFile),
    Spdx = get_vendor_dependencies(Sbom),
    file:write_file(SbomFile, json:format(Spdx)).

get_vendor_dependencies(#{~"packages" := Packages}=Spdx) ->
    AppPackages = create_otp_app_packages(Spdx),
    VendorPackages = create_otp_vendor_packages(Spdx),

    VendorPackageIds = lists:map(fun (#{~"SPDXID" := Id}) -> Id end, VendorPackages),
    OTPPackageIds = lists:map(fun (#{~"SPDXID" := Id}) -> Id end, AppPackages),
    Packages1 = lists:filter(fun (#{~"SPDXID" := Id}) ->
                                     lists:member(Id, VendorPackageIds) andalso not lists:member(Id, OTPPackageIds)
                             end, Packages),
    Spdx#{~"packages" := Packages1}.


sbom_otp(#{sbom_file  := SbomFile, write_to_file := Write, input_file := Input}) ->
    Sbom = decode(SbomFile),
    ScanResults = decode(Input),
    Spdx = improve_sbom_with_info(Sbom, ScanResults),
    case Write of
        true ->
            file:write_file(SbomFile, json:format(Spdx));
            %% Should we not overwritte the given file?
            %% file:write_file("otp.spdx.json", json:format(Spdx));
        false ->
            {ok, Spdx}
    end.

-spec improve_sbom_with_info(Sbom :: map(), ScanResults :: map()) -> Result :: map().
improve_sbom_with_info(Sbom, ScanResults) ->
    FixFuns = sbom_fixing_functions(ScanResults),
    Spdx = lists:foldl(fun ({Fun, Data}, Acc) -> Fun(Data, Acc) end, Sbom, FixFuns),
    package_by_app(Spdx).

sbom_fixing_functions(ScanResults) ->
    Licenses = path_to_license(ScanResults),
    Copyrights = path_to_copyright(ScanResults),
    [{fun fix_project_name/2, ?spdxref_project_name},
     {fun fix_name/2, ?spdx_project_name},
     {fun fix_creators_tooling/2, {?spdx_creators_tooling, ScanResults}},
     {fun fix_supplier/2, ?spdx_supplier},
     {fun fix_download_location/2, ?spdx_download_location},
     {fun fix_project_package_license/2, {Licenses, Copyrights}},
     {fun fix_project_package_version/2, 'OTP_VERSION'},
     {fun fix_has_extracted_license_info/2, extracted_license_info()},
     {fun fix_project_purl/2, ?spdx_project_purl},
     {fun fix_beam_licenses/2, {Licenses, Copyrights}}
    ].

fix_project_name(ProjectName, #{ ~"documentDescribes" := [ ProjectName0 ],
                                 ~"packages" := Packages}=Sbom) ->
    Packages1 = [begin
                     case maps:get(~"SPDXID", Package) of
                         ProjectName0 ->
                             Package#{~"SPDXID" := ProjectName};
                         _ ->
                             Package
                     end
                 end || Package <- Packages],
    Sbom#{ ~"documentDescribes" := [ ProjectName ], ~"packages" := Packages1}.

fix_name(Name, Sbom) ->
    Sbom#{ ~"name" => Name}.

fix_creators_tooling({Tool, #{~"repository" := #{~"vcs_processed" := #{~"revision" := Version}}}},
                      #{ ~"creationInfo" := #{~"creators" := [ORT | _]}=Creators}=Sbom) ->
    SHA = string:trim(<<".sha.", Version/binary>>),
    Sbom#{~"creationInfo" := Creators#{ ~"creators" := [ORT, <<Tool/binary, SHA/binary>>]}}.

fix_supplier(_Name, #{~"packages" := [ ] }=Sbom) ->
    io:format("[warn] no packages available!~n"),
    Sbom;
fix_supplier(Name, #{~"packages" := [_ | _]=Packages }=Sbom) ->
    Sbom#{~"packages" := [maps:update_with(~"supplier", fun(_) -> Name end, Name, Package) || Package <- Packages]}.

fix_download_location(_Url, #{~"packages" := [ ] }=Sbom) ->
    io:format("[warn] no packages available!~n"),
    Sbom;
fix_download_location(Url, #{~"packages" := [ _ | _ ]=Packages }=Sbom) ->
    PackagesUpdated = [ Package#{~"downloadLocation" := Url } || Package <- Packages],
    Sbom#{~"packages" := PackagesUpdated}.

fix_project_package_license(_, #{ ~"documentDescribes" := [RootProject],
                                  ~"packages" := Packages}=Spdx) ->
    Packages1= [case maps:get(~"SPDXID", Package) of
                    RootProject ->
                        Licenses = remove_invalid_spdx_licenses(maps:get(~"licenseDeclared", Package)),
                        Package#{ ~"homepage" := ~"https://www.erlang.org",
                                  ~"licenseConcluded" := binary:join(Licenses, ~" AND ")};
                    _ ->
                        Package
                end || Package <- Packages],
    Spdx#{~"packages" := Packages1}.

remove_invalid_spdx_licenses(Licenses) when is_list(Licenses) ->
    lists:foldl(fun (L, Acc) ->
                        remove_invalid_spdx_licenses(L) ++ Acc
                end, [], Licenses);
remove_invalid_spdx_licenses(Licenses) when is_binary(Licenses) ->
    lists:filter(fun (~"NONE") -> false;
                     (~"NOASSERTION") -> false;
                     (_) -> true
                 end, string:split(Licenses, ~" AND ", all)).

fix_project_package_version(_, #{ ~"documentDescribes" := [RootProject],
                                  ~"packages" := Packages}=Spdx) ->
    OtpVersion = get_otp_version(),
    Packages1= [case maps:get(~"SPDXID", Package) of
                    RootProject ->
                        Package#{ ~"versionInfo" := OtpVersion };
                    _ ->
                        Package
                end || Package <- Packages],
    Spdx#{~"packages" := Packages1}.

get_otp_version() ->
    {ok, Content} = file:read_file(?otp_version),
    string:trim(Content).

fix_project_purl(#{~"referenceLocator" := RefLoc}=Purl, #{ ~"documentDescribes" := [RootProject],
                          ~"packages" := Packages}=Spdx) ->
    Packages1= [case maps:get(~"SPDXID", Package) of
                    RootProject ->
                        VersionInfo = maps:get(~"versionInfo", Package),
                        Purl1 = Purl#{~"referenceLocator" := <<RefLoc/binary, "@OTP-", VersionInfo/binary>>},
                        Package#{ ~"externalRefs" => [Purl1]};
                    _ ->
                        Package
                end || Package <- Packages],
    Spdx#{~"packages" := Packages1}.

otp_purl(Name, VersionInfo) ->
    Metadata = ?spdx_purl_meta_data,
    <<"pkg:otp/", Name/binary, "@", VersionInfo/binary, Metadata/binary>>.

fix_has_extracted_license_info(MissingLicenses, #{~"hasExtractedLicensingInfos" := LicenseInfos,
                                                   ~"packages" := Packages,
                                                  ~"documentDescribes" := [RootProject]}=Spdx) ->
    ExtractedLicenses = [maps:get(~"licenseId", ExtractedLicense) || ExtractedLicense <- LicenseInfos ],
    MissingExtractedLicenses =
        lists:foldl(fun (Package, Acc) ->
                            case maps:get(~"SPDXID", Package) of
                                RootProject ->
                                    %% list of SPDX identifier
                                    InfoFromFiles = maps:get(~"licenseInfoFromFiles", Package),

                                    %% Licenses is a list of tuples (Spdx Id, License Text)
                                    Licenses = lists:filter(
                                                 fun (License) ->
                                                         %% License must be used, and not already extracted.
                                                         %% this only makes sense for LicenseRef-XXXX
                                                         lists:member(element(1, License), InfoFromFiles) andalso
                                                             not lists:member(element(1, License), ExtractedLicenses)
                                                 end, MissingLicenses),
                                    Licenses ++ Acc;
                              _ ->
                                  Acc
                          end
                  end, [], Packages),
    AddAllExtractedLicenses = lists:foldl(fun ({K, V}, Acc) ->
                                                  [#{~"extractedText" => V, ~"licenseId" => K} | Acc]
                                          end, LicenseInfos, MissingExtractedLicenses),
    Spdx#{~"hasExtractedLicensingInfos" := AddAllExtractedLicenses}.

-spec create_externalRef_purl(Desc :: binary(), Purl :: binary()) -> map().
create_externalRef_purl(Description, Purl) ->
    #{ ~"comment" => Description,
       ~"referenceCategory" => ~"PACKAGE-MANAGER",
       ~"referenceLocator" => Purl,
       ~"referenceType" => ~"purl"}.

%% re-populate licenses to .beam files from their .erl files
%% e.g., the lists.beam file should have the same license as lists.erl
fix_beam_licenses(_LicensesAndCopyrights, #{ ~"packages" := []}=Sbom) ->
    io:format("[warn] no packages available!~n"),
    Sbom;
fix_beam_licenses(LicensesAndCopyrights,
                  #{ ~"files"   := Files}=Sbom) ->

    Files1= lists:map(
              fun (SPDX) ->
                      %% Adds license and copyright from .erl or .hrl file to its .beam equivalent
                      case SPDX of
                          #{~"fileName" := <<"lib/stdlib/uc_spec/", _Filename/binary>>,
                            ~"licenseInfoInFiles" := [License]}  when License =/= ~"NONE", License =/= ~"NOASSERTION"->
                              files_have_no_license(SPDX#{~"licenseConcluded" := License});

                          #{~"fileName" := ~"bootstrap/lib/stdlib/ebin/erl_parse.beam"} ->
                              %% beam file auto-generated from grammar file
                              Spdx1 = fix_beam_spdx_license(~"lib/stdlib/src/erl_parse.yrl", LicensesAndCopyrights, SPDX),
                              Spdx2 = files_have_no_license(Spdx1),
                              add_license_comment(Spdx2);

                          #{~"fileName" := ~"bootstrap/lib/stdlib/ebin/unicode_util.beam"} ->
                              %% follows from otp/lib/stdlib/uc_spec/README-UPDATE.txt
                              Spdx1 = files_have_no_license(SPDX#{~"licenseConcluded" := ~"Unicode-3.0 AND Apache-2.0"}),
                              add_license_comment(Spdx1);

                          #{~"fileName" := Filename} when
                                Filename =:= ~"erts/emulator/zstd/COPYING";
                                Filename =:= ~"erts/emulator/zstd/LICENSE";
                                Filename =:= ~"erts/emulator/ryu/LICENSE-Apache2";
                                Filename =:= ~"erts/emulator/ryu/LICENSE-Boost";
                                Filename =:= ~"lib/eldap/LICENSE";
                                Filename =:= ~"erts/lib_src/yielding_c_fun/test/examples/sha256_erlang_nif/c_src/sha-2/LICENSE";
                                Filename =:= ~"erts/lib_src/yielding_c_fun/test/examples/sha256_erlang_nif/LICENSE" ->
                              %% license files have comment stating they are license files.
                              SPDX#{~"comment" => ~"license file"};

                          #{~"fileName" := <<"FILE-HEADERS/", Filename/binary>>} when Filename =/= ~"README.md" ->
                              %% license files have comment stating they are license files.
                              %% this cannot be encoded in .ort.yml as it does not allow to add comments
                              SPDX#{~"comment" => ~"license file"};  % TODO: remove this later ~"licenseInfoInFiles" := [~"NOASSERTION"]};

                          #{~"fileName" := <<"LICENSES/", _Filename/binary>>} ->
                              %% license files have comment stating they are license files.
                              SPDX#{~"comment" => ~"license file", ~"licenseInfoInFiles" := [~"NOASSERTION"]};

                          #{~"fileName" := Filename} ->
                              case bootstrap_mappings(Filename) of
                                  {error, not_beam_file} ->
                                      fix_spdx_license(SPDX);
                                  {Path, Filename1} ->
                                      case binary:split(Filename1, ~".beam") of
                                          [File, _] ->
                                              Spdx1 = fix_beam_spdx_license(Path, File, LicensesAndCopyrights, SPDX),
                                              Spdx2 = files_have_no_license(Spdx1),
                                              add_license_comment(Spdx2);
                                          _ ->
                                              fix_spdx_license(SPDX)
                                      end
                              end
                          end
              end, Files),
    Sbom#{ ~"files" := Files1}.

bootstrap_mappings(<<"bootstrap/lib/compiler/ebin/", Filename/binary>>) -> {~"lib/compiler/src/", Filename};
bootstrap_mappings(<<"bootstrap/lib/kernel/ebin/",Filename/binary>>) -> {<<"lib/kernel/src/">>, Filename};
bootstrap_mappings(<<"bootstrap/lib/kernel/include/",Filename/binary>>) -> {<<"lib/kernel/include/">>, Filename};
bootstrap_mappings(<<"bootstrap/lib/stdlib/ebin/",Filename/binary>>) -> {<<"lib/stdlib/src/">>, Filename};
bootstrap_mappings(<<"erts/preloaded/ebin/",Filename/binary>>) -> {<<"erts/preloaded/src/">>, Filename};
bootstrap_mappings(_Other) ->
    {error, not_beam_file}.


%% fixes spdx license of beam files
fix_beam_spdx_license(Path, {Licenses, Copyrights}, SPDX) ->
    License = maps:get(Path, Licenses, ~"NOASSERTION"),
    Copyright = maps:get(Path, Copyrights, ~"NOASSERTION"),
    fix_spdx_license(SPDX#{ ~"copyrightText" := Copyright, ~"licenseConcluded" := License }).

fix_beam_spdx_license(Path, File, LicensesAndCopyrights, SPDX) when is_binary(Path),
                                                                    is_binary(File) ->
    Spdx0 = fix_beam_spdx_license(<<Path/binary, File/binary, ".erl">>, LicensesAndCopyrights, SPDX),
    case maps:get(~"licenseConcluded", Spdx0) of
        ~"NOASSERTION" ->
            fix_beam_spdx_license(<<Path/binary, File/binary, ".hrl">>, LicensesAndCopyrights, Spdx0);
        _ ->
            Spdx0
    end.

files_have_no_license(Spdx) ->
    Spdx#{~"licenseInfoInFiles" := [~"NONE"]}.

none_to_noassertion(~"NONE") ->
    ~"NOASSERTION";
none_to_noassertion(X) ->
    X.

add_license_comment(#{~"licenseConcluded" := Concluded,
                     ~"licenseInfoInFiles" := [License]}=Spdx)
  when (Concluded =:= ~"NOASSERTION" orelse Concluded =:= ~"NONE") andalso License =/= Concluded ->
    Spdx#{~"licenseComments" => ~"BEAM files preserve their *.erl license"};
add_license_comment(Spdx) ->
    Spdx.


%% fixes spdx license of non-beam files
fix_spdx_license(#{~"licenseInfoInFiles" := [LicenseInFile],
                   ~"licenseConcluded" := License,
                   ~"copyrightText" := C}=SPDX) ->
    License1 = case License of
                   ~"NONE" -> LicenseInFile;
                   ~"NOASSERTION" -> LicenseInFile;
                   Other -> Other
               end,
    ConcludedLicense = none_to_noassertion(License1),
    SPDX#{ ~"licenseConcluded" := ConcludedLicense,
           ~"copyrightText" := none_to_noassertion(C) };
fix_spdx_license(#{~"licenseInfoInFiles" := Licenses}=SPDX) when length(Licenses) > 1 ->
    Licenses1 = lists:map(fun erlang:binary_to_list/1, Licenses),
    LicensesBin = erlang:list_to_binary(lists:join(" AND ", Licenses1)),
    fix_spdx_license(SPDX#{ ~"licenseInfoInFiles" := [LicensesBin] });
fix_spdx_license(#{~"copyrightText" := C}=SPDX) ->
    SPDX#{ ~"copyrightText" := none_to_noassertion(C)}.

%% Given an input file, returns a mapping of
%% #{filepath => license} for each file path towards its license.
-spec path_to_license(Input :: map()) -> #{Path :: binary() => License :: binary()}.
path_to_license(Input) ->
    match_path_to(Input, fun group_by_licenses/3).

-spec path_to_copyright(Input :: map()) -> #{Path :: binary() => License :: binary()}.
path_to_copyright(Input) ->
    match_path_to(Input, fun group_by_copyrights/3).

-spec match_path_to(Input :: map(), GroupFun :: fun()) -> #{ Path :: binary() => Result :: binary() }.
match_path_to(Json, GroupFun) ->
    Exclude = true,
    Curations = false,
    GroupedResult = GroupFun(Json, Exclude, Curations),
    maps:fold(fun (K, Vs, Acc) ->
                      maps:merge(maps:from_keys(Vs, K), Acc)
              end, #{}, GroupedResult).

%%
%% Explore command
%%
classify_license(#{output_file := Output,
                   input_file := Filename,
                   exclude := ApplyExclude,
                   curations := ApplyCuration}) ->
    Json = decode(Filename),
    R = group_by_licenses(Json, ApplyExclude, ApplyCuration),
    ok = file:write_file(Output, json:encode(R)).

classify_path_license_copyright(#{output_file := Output,
                     input_file := Filename,
                     base_file  := LicenseFileGroup}) ->
    Copyrights = classify_copyright_result(Filename),
    Licenses = expand_license_result(LicenseFileGroup),
    Files = lists:sort(lists:uniq(maps:keys(Copyrights) ++ maps:keys(Licenses))),
    X = lists:foldl(fun (Path, Acc) ->
                          Copyright = maps:get(Path, Copyrights, ~"NONE"),
                          License = maps:get(Path, Licenses, ~"NONE"),
                          Acc#{Path => #{ ~"Copyright" => Copyright, ~"License" => License}}
                    end, #{}, Files),
    ok = file:write_file(Output, json:encode(X)).

expand_license_result(Filename) ->
    Json = decode(Filename),
    maps:fold(fun (License, Paths, Acc) ->
                      maps:merge(Acc, maps:from_list([{Path, License} || Path <- Paths]))
              end, #{}, Json).

classify_copyright_result(Filename) ->
    Json = decode(Filename),
    Copyrights = copyrights(scan_results(Json)),
    lists:foldl(fun (Copyright, Acc) ->
                        #{<<"statement">> := CopyrightSt, <<"location">> := Location} = Copyright,
                        #{<<"path">> := Path, <<"start_line">> := _StartLine, <<"end_line">> := _EndLine} = Location,
                        Acc#{Path => CopyrightSt}
                    end, #{}, Copyrights).

-spec group_by_licenses(map(), boolean(), boolean()) -> #{License :: binary() => [Path :: binary()]}.
group_by_licenses(Json, ApplyExclude, ApplyCuration) ->
    Excludes = apply_excludes(Json, ApplyExclude),
    Curations = apply_curations(Json, ApplyCuration),

    Licenses = licenses(scan_results(Json)),
    lists:foldl(fun (License, Acc) ->
                            group_by_license(Excludes, Curations, License, Acc)
                    end, #{}, Licenses).

group_by_copyrights(Json, ApplyExclude, _ApplyCuration) ->
    Excludes = apply_excludes(Json, ApplyExclude),
    Copyrights = copyrights(scan_results(Json)),
    lists:foldl(fun (Copyright, Acc) ->
                            group_by_copyright(Excludes, Copyright, Acc)
                    end, #{}, Copyrights).


apply_excludes(Json, ApplyExclude) ->
    onlyif([], ApplyExclude, fun () -> convert_excludes(excludes(Json)) end).

apply_curations(Json, ApplyCuration) ->
    onlyif([], ApplyCuration, fun () -> curations(Json) end).

diff(#{input_file := InputFile, base_file := BaseFile, output_file := Output}) ->
    Input = decode(InputFile),
    Base = decode(BaseFile),
    KeyList = maps:keys(Input) ++ maps:keys(Base),
    KeySet = sets:from_list(KeyList),
    Data = sets:fold(fun(Key, Acc) -> set_difference(Key, Input, Base, Acc) end, #{}, KeySet),
    file:write_file(Output, json:encode(Data)).

detect_no_license(#{input_file := InputFile,
                    output_file := OutputFile,
                    exclude := ApplyExcludes}) ->
    Input = decode(InputFile),
    SortedResult = compute_unlicense_files(Input, ApplyExcludes),
    file:write_file(OutputFile, json:encode(SortedResult)).

compute_unlicense_files(Input, ApplyExcludes) ->
    Licenses = licenses(scan_results(Input)),

    PathsWithLicense =
        lists:foldl(fun (#{<<"location">> := #{<<"path">> := Path}}, Acc) ->
                            sets:add_element(Path, Acc)
                    end, sets:new(), Licenses),

    %% Get all files, incluiding those without license
    Files = files_from_scanner(Input),
    AllPaths =
        lists:foldl(fun (#{<<"path">> := Path}, Acc) ->
                            sets:add_element(Path, Acc)
                    end, sets:new(), Files),

    %% Paths without license
    PathsWithoutLicense = sets:to_list(sets:subtract(AllPaths, PathsWithLicense)),

    %% Excluded files that should be ignored
    Excludes = excludes(Input),
    ExcludeRegex = onlyif([], ApplyExcludes, fun () -> convert_excludes(Excludes) end),
    Result = lists:foldl(fun(Path, Acc) ->
                                 case exclude_path(Path, ExcludeRegex) of
                                     true ->
                                         Acc;
                                     false ->
                                         [Path | Acc]
                                 end
                         end, [], PathsWithoutLicense),
    lists:sort(Result).

check_no_license(#{input_file := InputFile,
                   base_file := BaseFile,
                   exclude := ApplyExcludes,
                   output_file := OutputFile}) ->
    UnlicenseNew = compute_unlicense_files(decode(InputFile), ApplyExcludes),
    Unlicense = decode(BaseFile),
    UnlicenseSet = sets:from_list(Unlicense),
    UnlicenseNewSet =  sets:from_list(UnlicenseNew),
    Result = sets:to_list(sets:subtract(UnlicenseNewSet, UnlicenseSet)),
    file:write_file(OutputFile, json:encode(Result)).


%%
%% Helper functions
%%

excludes(Input) ->
    try
        #{<<"repository">> :=
              #{<<"config">> :=
                    #{<<"excludes">> := #{<<"paths">> := Excludes}}}} = Input,
        Excludes
    catch
        _:_ ->
            []
    end.


curations(Input) ->
    #{<<"repository">> :=
          #{<<"config">> :=
                #{<<"curations">> := #{<<"license_findings">> := Curations}}}} = Input,
    Curations.

scan_results(Input) ->
    #{<<"scanner">> := #{<<"scan_results">> := ScanResults}} = Input,
    ScanResult = hd(ScanResults),
    NewSummary =
        lists:foldl(fun(#{ ~"summary" := #{ ~"licenses" := Licenses, ~"copyrights" := Copyrights}}, Acc) ->
            Acc#{ ~"licenses" := Licenses ++ maps:get(~"licenses", Acc),
                ~"copyrights" := Copyrights ++ maps:get(~"copyrights", Acc) }
        end, maps:get(~"summary",ScanResult), tl(ScanResults)),
    ScanResult#{ ~"summary" := NewSummary }.

licenses(Input) ->
    #{<<"summary">> := #{<<"licenses">> := Licenses}} = Input,
    Licenses.

copyrights(Input) ->
    #{<<"summary">> := #{<<"copyrights">> := Copyrights}} = Input,
    Copyrights.


files_from_scanner(Input) ->
    #{<<"scanner">> := #{<<"files">> := [#{<<"files">> := Files}]}} = Input,
    Files.

set_difference(Key, Input, Base, Acc) ->
    InputValues = sets:from_list(maps:get(Key, Input, [])),
    BaseValues = sets:from_list(maps:get(Key, Base, [])),
    Additions = sets:subtract(InputValues, BaseValues),
    Deletions = sets:subtract(BaseValues, InputValues),
    Acc#{Key => #{addition => sets:to_list(Additions), deletions => sets:to_list(Deletions)}}.

onlyif(_Default, true, Command) -> Command();
onlyif(Default, false, _Command) -> Default.

decode(Filename) ->
    {ok, Bin} = file:read_file(Filename),
    json:decode(Bin).

decode_without_spdx_license(Filename) ->
    {ok, Bin} = file:read_file(Filename),

    %% remove comments
    Lines = string:split(Bin, "\n", all),
    Lines1 = lists:map(fun (Line) -> re:replace(Line, "^//.*", "", [global]) end, Lines),
    Bin1 = erlang:iolist_to_binary(Lines1),

    json:decode(Bin1).

group_by_license(ExcludeRegexes, Curations, License, Acc) ->
    #{<<"license">> := LicenseName, <<"location">> := Location} = License,
    #{<<"path">> := Path, <<"start_line">> := _StartLine, <<"end_line">> := _EndLine} = Location,
    maybe
        false ?= exclude_path(Path, ExcludeRegexes),
        LicenseName1 = curated_path_license(LicenseName, Path, Curations),
        case maps:get(LicenseName1, Acc, []) of
            [] ->
                Acc#{LicenseName1 => [Path]};
            Ls ->
                Ls1 = case lists:search(fun(X) -> X == Path end, Ls) of
                          false -> [Path | Ls];
                          _ -> Ls
                      end,
                Acc#{LicenseName1 => Ls1}
        end
    else
        _ ->
            Acc
    end.

group_by_copyright(ExcludeRegexes, Copyright, Acc) ->
    #{<<"statement">> := CopyrightSt, <<"location">> := Location} = Copyright,
    #{<<"path">> := Path, <<"start_line">> := _StartLine, <<"end_line">> := _EndLine} = Location,
    maybe
        false ?= exclude_path(Path, ExcludeRegexes),
        case maps:get(CopyrightSt, Acc, []) of
            [] ->
                Acc#{CopyrightSt => [Path]};
            Ls ->
                Ls1 = case lists:search(fun(X) -> X == Path end, Ls) of
                          false -> [Path | Ls];
                          _ -> Ls
                      end,
                Acc#{CopyrightSt => Ls1}
        end
    else
        _ ->
            Acc
    end.

convert_excludes(Excludes) ->
    lists:map(fun (#{<<"pattern">> := Pattern}) ->
                      Pattern1 = re:replace(Pattern, <<"\\.">>, <<"\\\\.">>, [global, {return, binary}]),
                      re:replace(Pattern1, <<"\\*\\*">>, <<".*">>, [global, {return, binary}])
              end, Excludes).

exclude_path(_Path, []) ->
    false;
exclude_path(Path, ExcludeRegexes) ->
    lists:any(fun (Regex) ->
                      case re:run(Path, Regex) of
                          {match, _} -> true;
                          _ -> false
                      end
              end, ExcludeRegexes).

curated_path_license(Name, _Path, []) -> Name;
curated_path_license(_Name, Path, [#{<<"path">> := Path}=Cur | _Curations]) ->
    maps:get(<<"concluded_license">>, Cur);
curated_path_license(Name, Path, [_Cur | Curations]) ->
    curated_path_license(Name, Path, Curations).

%% fixes the Spdx to split Spdx by app, and adds vendor dependencies
package_by_app(Spdx) ->
    %% add App packages, e.g., stdlib, erts, ssh, ssl
    AppSrcFiles = find_app_src_files("."),
    PackageTemplates = generate_spdx_mappings(AppSrcFiles),
    Packages = generate_spdx_packages(PackageTemplates, Spdx),
    AppPackages = lists:map(fun create_spdx_package/1, Packages),
    Spdx1 = add_packages(AppPackages, Spdx),
    Spdx2 = create_otp_relationships(Packages, PackageTemplates, Spdx1),

    %% create vendor packages
    VendorPackages = create_otp_vendor_packages(Spdx2),

    %% Remove possible duplicates of vendor packages
    {NewVendorPackages, Spdx3} = remove_duplicate_packages(VendorPackages, Spdx2),

    SpdxWithVendor = add_packages(NewVendorPackages, Spdx3),
    create_vendor_relations(NewVendorPackages, SpdxWithVendor).

create_otp_app_packages(Spdx) ->
    AppSrcFiles = find_app_src_files("."),
    PackageTemplates = generate_spdx_mappings(AppSrcFiles),
    Packages = generate_spdx_packages(PackageTemplates, Spdx),
    lists:map(fun create_spdx_package/1, Packages).

create_otp_vendor_packages(Spdx) ->
    VendorSrcFiles = find_vendor_src_files("."),
    VendorInfoPackage = generate_vendor_info_package(VendorSrcFiles),
    generate_spdx_vendor_packages(VendorInfoPackage, Spdx).

create_otp_relationships(Packages, PackageTemplates, Spdx) ->
    Spdx1 = create_package_relationships(Packages, Spdx),
    Spdx2 = create_depends_on_relationships(PackageTemplates, Spdx1),
    create_opt_depency_relationships(PackageTemplates, Spdx2).

-spec add_packages(Packages :: [spdx_package()], Spdx :: map()) -> SpdxResult :: map().
add_packages(AppPackages, Spdx) ->
    #{~"packages" := SpdxPackages}=Spdx1 = remove_package_files_from_project(Spdx, AppPackages),
    Spdx1#{~"packages" := SpdxPackages ++ AppPackages}.

%% Removes duplicate packages and adds a comment for existing vendor Packages in SPDX
%% it also remove files in top-level directories and they onyly exist  in vendor libraries
-spec remove_duplicate_packages(VendorPackages :: map(), Spdx2 :: map()) -> {ResultVendorPackages :: map(), SPDX :: map()}.
remove_duplicate_packages(VendorPackages, #{~"packages" := Packages}=Spdx) ->
    #{~"vendor" := Vendors, ~"app" := Apps} =
        lists:foldl(fun (#{~"SPDXID" := VendorId}=Vendor, #{~"vendor" := Vcc, ~"app" := Apc}=Acc) ->
                            case lists:search(fun (#{~"SPDXID" := Id}) -> VendorId == Id end, Packages) of
                                {value, P} ->
                                    Packages1 = Apc -- [P],
                                    Comment = maps:get(~"comment", P, <<>>),
                                    Comment1 =
                                        case Comment of
                                            <<>> ->
                                                ~"vendor package";
                                            _ ->
                                                <<Comment/binary, " vendor package">>
                                        end,
                                    Acc#{~"app" := [P#{~"comment" => Comment1} | Packages1]};
                                _ ->
                                    Acc#{~"vendor" := [Vendor | Vcc]}
                            end
                    end, #{~"vendor" => [], ~"app" => Packages}, VendorPackages),

    %%
    VendorFileIds = lists:flatten(lists:map(fun (#{~"hasFiles" := Fs}) -> Fs end, Vendors)),
    FixedApps = lists:map(fun (#{~"hasFiles" := SPDXIDs}=AppPackage) ->
                                  AppPackage#{~"hasFiles" := SPDXIDs -- VendorFileIds}
                          end, Apps),
    {Vendors, Spdx#{~"packages" := FixedApps}}.

%% project package contains `hasFiles` fields with all files.
%% remove all files included in other packages from project package.
%% there exists already a package relation between packages and project package.
remove_package_files_from_project(#{~"documentDescribes" := [ProjectPackageId],
                                    ~"packages" := Packages}=Spdx, AppPackages) ->
    [#{~"hasFiles" := FilesId}=ProjectPackage] = lists:filter(fun (#{~"SPDXID" := SPDXID}) -> SPDXID == ProjectPackageId end, Packages),
    AppFilesId = lists:foldl(fun (#{~"hasFiles" := Files}, Acc) -> Files ++ Acc end, [], AppPackages),
    ProjectPackage1 = ProjectPackage#{~"hasFiles" := FilesId -- AppFilesId},
    Spdx#{~"packages" := [ProjectPackage1 | Packages -- [ProjectPackage]]}.


-spec create_spdx_package(Package :: spdx_package()) -> map().
create_spdx_package(Pkg) ->
    SPDXID = Pkg#spdx_package.'SPDXID',
    VersionInfo= Pkg#spdx_package.'versionInfo',
    Name = Pkg#spdx_package.'name',
    CopyrightText = Pkg#spdx_package.'copyrightText',

    FilesAnalyzed = Pkg#spdx_package.'filesAnalyzed',
    HasFiles = Pkg#spdx_package.'hasFiles',
    Homepage = Pkg#spdx_package.'homepage',
    LicenseConcluded = Pkg#spdx_package.'licenseConcluded',
    LicenseDeclared = Pkg#spdx_package.'licenseDeclared',
    LicenseInfo = Pkg#spdx_package.'licenseInfoFromFiles',
    DownloadLocation = Pkg#spdx_package.'downloadLocation',
    PackageVerification = Pkg#spdx_package.'packageVerificationCode',
    PackageVerificationCodeValue = maps:get('packageVerificationCodeValue', PackageVerification),
    Supplier = Pkg#spdx_package.'supplier',
    Purl1 = case Pkg#spdx_package.'purl' of
               false -> [];
               _ -> Pkg#spdx_package.'purl'
           end,
    #{ ~"SPDXID" => SPDXID,
       ~"versionInfo" => VersionInfo,
       ~"name" => Name,
       ~"copyrightText" => CopyrightText,
       ~"filesAnalyzed" => FilesAnalyzed,
       ~"hasFiles" => HasFiles,
       ~"homepage" => Homepage,
       ~"licenseConcluded" => LicenseConcluded,
       ~"licenseDeclared" => LicenseDeclared,
       ~"licenseInfoFromFiles" => LicenseInfo,
       ~"downloadLocation" => DownloadLocation,
       ~"externalRefs" => Purl1,
       ~"packageVerificationCode" => #{~"packageVerificationCodeValue" => PackageVerificationCodeValue},
       ~"supplier" => Supplier
     }.

%% Example:
%% https://github.com/spdx/tools-java/blob/master/testResources/SPDXJSONExample-v2.2.spdx.json#L240-L275
create_package_relationships(Packages, Spdx) ->
    Relationships =
        lists:foldl(fun (Pkg, Acc) ->
                            {Key, Ls} = case Pkg#spdx_package.'relationships' of
                                            #{'PACKAGE_OF' := L } -> {'PACKAGE_OF', L};
                                            #{'TEST_OF' := L} -> {'TEST_OF', L};
                                            #{'DOCUMENTATION_OF' := L} -> {'DOCUMENTATION_OF', L}
                                        end,
                            lists:foldl(fun ({ElementId, RelatedElement}, Acc1) ->
                                              [create_spdx_relation(Key, ElementId, RelatedElement) | Acc1]
                                      end, Acc, Ls)
                    end, [], Packages),
    Spdx#{~"relationships" => Relationships}.

-spec create_depends_on_relationships(PackageMappings, Spdx) -> map() when
      PackageMappings :: #{AppName => {AppPath, app_info()}},
      AppName :: binary(),
      AppPath :: binary(),
      Spdx :: map().
create_depends_on_relationships(PackageTemplates, #{~"relationships" := Relationships}=Spdx) ->
    DependsOn =
        maps:fold(fun (PackageName, {_Path, AppInfo}, Acc) ->
                          DependsOnApps = lists:map(fun erlang:atom_to_binary/1, AppInfo#app_info.applications),
                          SpdxPackageName = generate_spdxid_name(PackageName),
                          Relations = [create_spdx_relation('DEPENDS_ON', SpdxPackageName, generate_spdxid_name(RelatedElement))
                                        || RelatedElement <- [~"erts" | DependsOnApps], generate_spdxid_name(RelatedElement) =/= SpdxPackageName],
                           Relations ++ Acc
                   end, [], PackageTemplates),
    Spdx#{~"relationships" := DependsOn ++ Relationships}.

create_opt_depency_relationships(PackageTemplates, #{~"relationships" := Relationships}=Spdx) ->
    DependsOn =
        maps:fold(fun (PackageName, {_Path, AppInfo}, Acc) ->
                          Optional = AppInfo#app_info.included_applications ++ AppInfo#app_info.optional_applications,
                          DependsOnApps = lists:map(fun erlang:atom_to_binary/1, Optional),
                          SpdxPackageName = generate_spdxid_name(PackageName),
                          Relations = [create_spdx_relation('OPTIONAL_DEPENDENCY_OF', generate_spdxid_name(RelatedElement), SpdxPackageName)
                                        || RelatedElement <- DependsOnApps, generate_spdxid_name(RelatedElement) =/= SpdxPackageName],
                           Relations ++ Acc
                   end, [], PackageTemplates),
    Spdx#{~"relationships" := DependsOn ++ Relationships}.


%% adds package of to packages within packages in OTP.
%% example: asmjit is a subpackage of erts
create_vendor_relations(NewVendorPackages, #{~"packages" := Packages, ~"relationships" := Relations}=SpdxWithVendor) ->
    VendorRelations =
        lists:map(fun (#{~"name" := _Name, ~"SPDXID" := ID}=_Vendor) ->
                          %% Get root relation to point to
                          App = case string:split(undo_spdxid_name(ID), ~"-", all) of
                                    [BaseApp, ~"test" | _] ->
                                        <<BaseApp/binary, "-test">>;
                                    [BaseApp, ~"-documentation" | _] ->
                                        <<BaseApp/binary, "-documentation">>;
                                    [BaseApp | _] ->
                                        BaseApp
                                end,
                          Pkgs = lists:filter(fun (#{~"name" := N}) -> App == generate_spdx_valid_name(N) end, Packages),
                          case Pkgs of
                              [#{~"SPDXID" := RootId}=_RootPackage] ->
                                  create_spdx_relation('PACKAGE_OF', ID, RootId);
                              [] ->
                                  %% Attach to root level package
                                  create_spdx_relation('PACKAGE_OF', ID, ?spdxref_project_name)
                              end
                  end, NewVendorPackages),
    SpdxWithVendor#{~"relationships" := Relations ++ VendorRelations}.

-spec create_spdx_relation('PACKAGE_OF' | 'DEPENDS_ON', SpdxId :: binary(), RelatedId :: binary()) -> map().
create_spdx_relation(Relation, ElementId, RelatedElement) ->
    #{~"spdxElementId" => ElementId,
      ~"relatedSpdxElement" => RelatedElement,
      ~"relationshipType" => Relation}.

-spec find_app_src_files(Folder :: string()) -> [string()].
find_app_src_files(Folder) ->
    S = os:cmd("find "++ Folder ++ " -regex .*.app.src | grep -v test | grep -v smoke-build | cut -d/ -f2-"),

    %% TODO: merge above and below command into a single command
    %% specific line to include common_test, if it were to exist
    SCommonTest = os:cmd("find " ++ Folder ++ " -regex .*.app.src | grep -v test_dir | grep common_test | cut -d/ -f2-"),
    lists:map(fun erlang:list_to_binary/1, string:split(S ++ SCommonTest, "\n", all)).

get_otp_apps_from_table() ->
    {ok, BinTable} = file:read_file("otp_versions.table"),
    {ok, ReleaseBin0} = file:read_file("OTP_VERSION"),
    ReleaseBin = string:trim(ReleaseBin0),
    OTPVersion = <<"OTP-", ReleaseBin/binary>>,

    Lines = string:split(BinTable, "\n", all),
    LineOTP = lists:filter(fun (Line) ->
                                   [V | _Rest] = string:split(Line, " "),
                                   V == OTPVersion
                           end, Lines),
    case LineOTP of
        [Line] ->
            Line1 = re:replace(Line, "(#|:)", "", [global]),
            Line2 = string:trim(re:replace(Line1, OTPVersion, "", [global])),

            AppsWithVersion = string:split(Line2, " ", all),
            AppsWithVersion1 = lists:filter(fun (Bin) ->
                                                    case Bin of
                                                        <<>> ->
                                                            false;
                                                        _ ->
                                                            true
                                                    end
                                            end, AppsWithVersion),
            lists:map(fun (App) ->
                              [Name, Version] = string:split(App, "-"),
                              {Name, Version}
                      end, AppsWithVersion1);
            %% lists:map(fun (App) -> iolist_to_binary(re:replace(App, "-.*", "", [global])) end, AppsWithVersion);
        [] ->
            [];
        _ ->
            io:format("ERROR, there cannot be multiple lines matching")
    end.

find_vendor_src_files(Folder) ->
    string:split(string:trim(os:cmd("find "++ Folder ++ " -name vendor.info")), "\n", all).

-spec generate_spdx_mappings(Path :: [binary()]) -> Result when
      Result :: #{AppName :: binary() => {AppPath :: binary(), AppInfo :: app_info()}}.
generate_spdx_mappings(AppSrcPath) ->
    lists:foldl(fun (AppSrcPath0, Acc) ->
                        DetectedPackages = build_package_location(AppSrcPath0),
                        maps:merge(Acc, DetectedPackages)
                end, #{}, AppSrcPath).

%% Read Path file and generate Json (map) following vendor.info specification
-spec generate_vendor_info_package(VendorSrcPath :: [file:name()]) -> map().
generate_vendor_info_package(VendorSrcPath) ->
    lists:flatmap(fun decode_without_spdx_license/1, VendorSrcPath).

-spec generate_spdx_vendor_packages(VendorInfoPackage :: map(), map()) -> map().
generate_spdx_vendor_packages(VendorInfoPackages, #{~"files" := SpdxFiles}=_SPDX) ->
    RemoveVendorInfoFields = [~"purl", ~"ID", ~"path", ~"update", ~"exclude", ~"sha"],
    lists:map(fun
                  (#{~"ID" := Id, ~"path" := [_ | _]=ExplicitFiles}=Package) when is_list(ExplicitFiles) ->
                      %% Deals with the cases of creating a package out of specific files
                      Paths = lists:map(fun cleanup_path/1, ExplicitFiles),
                      Package1 = maps:without(RemoveVendorInfoFields, Package),
                      Excludes = get_vendor_excludes(Package),

                      %% place files in SPDX in the corresponding package
                      Files = lists:filter(fun (#{~"fileName" := Filename}) ->
                                                   case lists:member(Filename, Paths) of
                                                       false -> false;
                                                       true -> not exclude_vendor_file(Filename, Excludes)
                                                   end
                                           end, SpdxFiles),

                      LicenseInfoInFiles = split_licenses_in_individual_parts(
                        lists:foldl(fun(#{~"licenseInfoInFiles" := Licenses}, Acc) ->
                                            Licenses ++ Acc
                                    end, [], Files)),

                      PackageVerificationCodeValue = generate_verification_code_value(Files),
                      ExternalRefs = generate_vendor_purl(Package),
                      Package1#{
                                ~"SPDXID" => generate_spdxid_name(Id),
                                ~"filesAnalyzed" => true,
                                ~"hasFiles" => lists:map(fun (#{~"SPDXID":=Id0}) -> Id0 end, Files),
                                ~"licenseConcluded" => ~"NOASSERTION",
                                ~"licenseInfoFromFiles" => lists:uniq(LicenseInfoInFiles),
                                ~"packageVerificationCode" => #{~"packageVerificationCodeValue" => PackageVerificationCodeValue},
                                ~"comment" => ~"vendor package",
                                ~"externalRefs" => ExternalRefs
                       };
                  (#{~"ID" := Id, ~"path" := DirtyPath}=Package) when is_binary(DirtyPath) ->
                      %% Deals with the case of creating a package out of a path
                      Path = ensure_trailing_slash(cleanup_path(DirtyPath)),
                      true = filelib:is_dir(DirtyPath),
                      Package1 = maps:without(RemoveVendorInfoFields, Package),
                      Excludes = get_vendor_excludes(Package),

                      %% place files in SPDX in the corresponding package
                      Files = lists:filter(fun (#{~"fileName" := Filename}) ->
                                                   case string:prefix(Filename, Path) of
                                                       nomatch -> false;
                                                       _ -> not exclude_vendor_file(Filename, Excludes)
                                                   end
                                           end, SpdxFiles),
                      LicenseInfoInFiles = split_licenses_in_individual_parts(
                        lists:foldl(fun(#{~"licenseInfoInFiles" := Licenses}, Acc) ->
                                            Licenses ++ Acc
                                    end, [], Files)),

                      PackageVerificationCodeValue = generate_verification_code_value(Files),
                      ExternalRefs = generate_vendor_purl(Package),
                      Package1#{
                                ~"SPDXID" => generate_spdxid_name(Id),
                                ~"filesAnalyzed" => true,
                                ~"hasFiles" => lists:map(fun (#{~"SPDXID":=Id0}) -> Id0 end, Files),
                                ~"licenseConcluded" => ~"NOASSERTION",
                                ~"licenseInfoFromFiles" => lists:uniq(LicenseInfoInFiles),
                                ~"packageVerificationCode" => #{~"packageVerificationCodeValue" => PackageVerificationCodeValue},
                                ~"comment" => ~"vendor package",
                                ~"externalRefs" => ExternalRefs
                       }
              end, VendorInfoPackages).

get_vendor_excludes(Package) ->
    lists:map(fun (Exclude) ->
                      CleanExclude = cleanup_path(Exclude),
                      case filelib:is_dir(Exclude) of
                          true ->
                              {dir, ensure_trailing_slash(CleanExclude)};
                          false ->
                              true = filelib:is_regular(Exclude),
                              {file, CleanExclude}
                      end
              end, maps:get(~"exclude", Package, [])).

exclude_vendor_file(Filename, Excludes) ->
    lists:any(fun ({file, ExcludeFile}) ->
                      string:equal(Filename, ExcludeFile);
                  ({dir, ExcludeDir}) ->
                      case string:prefix(Filename, ExcludeDir) of
                          nomatch -> false;
                          _ -> true
                      end
              end, Excludes).

ensure_trailing_slash(Path) ->
    [string:trim(Path, trailing, "/"), $/].

generate_vendor_purl(Package) ->
    Description = maps:get(~"description", Package, ""),
    Vsn = maps:get(~"versionInfo", Package, false),
    Purl = maps:get(~"purl", Package, false),

    case {Purl, Vsn}  of
        {false, _} ->
            [];
        {Purl, false} ->
            [create_externalRef_purl(Description, Purl)];
        {Purl, Vsn} ->
            [create_externalRef_purl(Description, <<Purl/binary, "@", Vsn/binary>>)]
    end.

osv_scan(#{version := <<"maint">>}=Opt) ->
    VersionNumber = erlang:list_to_binary(string:trim(os:cmd("cat OTP_VERSION | cut -d. -f1"))),
    osv_scan(Opt#{version := <<"maint-", VersionNumber/binary>>});
osv_scan(#{version := Version,
           fail_if_cve := FailIfCVEFound}) ->
    application:ensure_all_started([ssl, inets]),
    _ = valid_scan_branches(Version),
    OSVQuery = vendor_by_version(Version),

    io:format("[OSV] Information sent~n~s~n", [json:format(OSVQuery)]),

    OSV = json:encode(OSVQuery),

    Format = "application/x-www-form-urlencoded",
    URI = "https://api.osv.dev/v1/querybatch",
    Content = {URI, [], Format, OSV},
    Result = httpc:request(post, Content, [], []),
    Vulns =
        case Result of
            {ok,{{_, 200,_}, _Headers, Body}} ->
                #{~"results" := OSVResults} = json:decode(erlang:list_to_binary(Body)),
                [{NameVersion, [Id || #{~"id" := Id} <- Ids]} ||
                    NameVersion <- osv_names(OSVQuery) && #{~"vulns" := Ids} <- OSVResults];
            {error, Error} ->
                {error, [URI, Error]}
        end,

    %% Substract from Vulns the OpenVex statements that dealt with them
    %% Result Vulns1 are vulnerabilities not yet covered in OpenVex statements
    Vulns1 = ignore_vex_cves(Version, Vulns),

    %% vulnerability reporting can fail if new issues appear
    FormattedVulns = format_vulnerabilities(Vulns1),
    case FailIfCVEFound of
        false ->
            report_vulnerabilities(FormattedVulns);
        true ->
            case Vulns1 of
                [] ->
                    report_vulnerabilities(FormattedVulns);
                _ ->
                    Failure =
                        """
                        **Vulnerability Detected**

                        The following CVEs must be checked in OpenVex statements for ~s:
                        ~s

                        Please follow instructions on how to do this from:
                          https://github.com/erlang/otp/blob/master/HOWTO/SBOM.md#vex
                        """,
                    create_or_update_gh_issue(Version, Failure, FormattedVulns),
                    fail(Failure, [Version, FormattedVulns])
            end
    end.

create_or_update_gh_issue(Version, BodyText, Vulns) ->
    VersionS = erlang:binary_to_list(Version),
    Cmd = "gh api -H \"Accept: application/vnd.github+json\" -H \"X-GitHub-Api-Version: 2022-11-28\" ",
    SearchCmd =
        io_lib:format("/search/issues?q=repo:~s+in:title+~s+~s+is:issue+is:open",
                      [?GH_ACCOUNT, VersionS, ?FOUND_VENDOR_VULNERABILITY]),

    io:format("Query GH API~n~s~n~n", [Cmd ++ SearchCmd]),
    RawResponse = cmd(Cmd ++ SearchCmd),
    Bin = unicode:characters_to_binary(RawResponse),
    #{~"total_count" := Count} = json:decode(Bin),
    FormattedBody = io_lib:format(BodyText, [Version, Vulns]),
    case Count of
        0 ->
            create_gh_issue(VersionS, ?FOUND_VENDOR_VULNERABILITY_TITLE, FormattedBody);
        _ ->
            ok
    end.

create_gh_issue(Version, Title, BodyText) ->
    Create = io_lib:format("gh issue create -t \"[~s] ~s\" -b \"~s\" -R ~s", [Version, Title, BodyText, ?GH_ACCOUNT]),
    io:format("GH Create Ticket with title '[~s] ~s'~n~s~n~n", [Version, Title, Create]),
    _ = cmd(Create),
    ok.

ignore_vex_cves(Branch, Vulns) ->
    OpenVex = download_otp_openvex_file(Branch),
    OpenVex1 = format_vex_statements(OpenVex),

    case OpenVex1 of
        [] ->
            [];
        _ when is_list(OpenVex1) ->
            io:format("Ignoring vulnerabilities already present in OpenVex file.~n~n")
    end,
    lists:foldl(fun({{Purl, _CommitId}=Package, CVEs}, Acc) ->
                        %% Ignore commit id when an OpenVEX statement exists.
                        %% OSV will report a vulnerability as long as Erlang/OTP does not
                        %% update its vendor.info file for openssl. we can only do this
                        %% when we actually vendor a different version of openssl, thus
                        %% the commit ids do not match. instead of basing vendor CVE checks
                        %% on commit id, if OTP adds an OpenVEX statement in which it claims
                        %% that there is no vulnerability, then there is no vulnerability.
                        %% If there is a vulnerability, then OTP must update the vendor file
                        %% to remove the vulnerability.
                        CVEsMatches = lists:filtermap(fun ({{PurlX, _}, CVEList}) ->
                                                              case string:lowercase(Purl) == string:lowercase(PurlX) of
                                                                  true ->
                                                                      {true, CVEList};
                                                                  false ->
                                                                      false
                                                              end;
                                                          (_) ->
                                                              false
                                                      end, OpenVex1),
                        case CVEs -- lists:flatten(CVEsMatches) of
                            [] ->
                                Acc;
                            Ls ->
                                [{Package, Ls} | Acc]
                        end
                end, [], Vulns).

format_vex_statements(OpenVex) ->
    Stmts = maps:get(~"statements", OpenVex, []),
    lists:foldl(fun (#{~"vulnerability" := #{~"name":=Name},
                       ~"products" := Products}, Acc) ->
                        Result =
                            lists:map(fun (#{~"@id" := <<"pkg:github/", Package/binary>>}) ->
                                              {PkgName, VersionPart} = string:take(Package, "@", true, leading),
                                              <<"@", Version/binary>> = VersionPart,
                                              {{<<"github.com/", PkgName/binary>>, Version}, [Name]};
                                          (_) ->
                                              Acc
                                      end, Products),
                        Result ++ Acc
              end, [], Stmts).

read_openvex_file(Branch) ->
    _ = create_dir(?VexPath),
    OpenVexPath = path_to_openvex_filename(Branch),
    OpenVexStr = erlang:binary_to_list(OpenVexPath),
    decode(OpenVexStr).

-spec download_otp_openvex_file(Branch :: binary()) -> Json :: map() | EmptyMap :: #{} | no_return().
download_otp_openvex_file(Branch) ->
    _ = create_dir(?VexPath),
    OpenVexPath = path_to_openvex_filename(Branch),
    OpenVexStr = erlang:binary_to_list(OpenVexPath),
    GithubURI = get_gh_download_uri(OpenVexStr),

    io:format("Checking OpenVex statements in '~s' from~n'~s'...~n", [OpenVexPath, GithubURI]),

    ValidURI = "curl -I -Lj --silent " ++ GithubURI ++ " | head -n1 | cut -d' ' -f2",
    case string:trim(os:cmd(ValidURI)) of
        "200" ->
            %% Overrides existing file.
            io:format("OpenVex file found.~n~n"),
            Command = "curl -LJ " ++ GithubURI ++ " --output " ++ OpenVexStr,
            io:format("Proceed to download:~n~s~n~n", [Command]),
            os:cmd(Command, #{ exception_on_failure => true }),
            decode(OpenVexStr);
        E ->
            io:format("[~p] No OpenVex statements found for file '~s'.~n~n", [E, OpenVexStr]),
            #{}
    end.

-spec get_gh_download_uri(String :: list()) -> String :: list().
get_gh_download_uri(File) ->
    ?OTP_GH_URI ++ File.

-spec create_dir(DirName :: binary()) -> ok | no_return().
create_dir(DirName) ->
    case file:make_dir(DirName) of
        Result when Result == ok;
                    Result == {error, eexist} ->
            io:format("Directory ~s created successfully.~n", [DirName]);
        {error, Reason} ->
            fail("Failed to create directory ~s: ~p~n", [DirName, Reason])
    end.

-spec path_to_openvex_filename(Branch :: binary()) -> Path :: binary().
path_to_openvex_filename(Branch) ->
    _ = valid_scan_branches(Branch),
    Version = maint_to_otp_conversion(Branch),
    vex_path(Version).

maint_to_otp_conversion(Branch) ->
    case Branch of
        ~"master" ->
            %% Master corresponds to possible patched versions of OTP_VERSION-1.
            BinVersionNumber = erlang:list_to_binary(string:trim(os:cmd("cat OTP_VERSION | cut -d. -f1"))),
            <<"otp-", BinVersionNumber/binary>>;
        <<"maint-", Vers/binary>> ->
            <<"otp-", Vers/binary>>;
        <<"maint">> ->
            BinVersionNumber = erlang:list_to_binary(string:trim(os:cmd("cat OTP_VERSION | cut -d. -f1"))),
            <<"otp-", BinVersionNumber/binary>>;
        <<"otp-", _Vers/binary>>=OTP ->
            OTP
    end.

-spec valid_scan_branches(Branch :: binary()) -> ok | no_return().
valid_scan_branches(Branch) ->
    case Branch of
        ~"master" ->
            ok;
        <<"maint-", _Vers/binary>> ->
            ok;
        <<"otp-", _Vers/binary>> ->
            ok;
        _ ->
            fail("[ERROR] Valid branch names are `master` or `maint-XX`.~n'~s' is neither of them", [Branch])
    end.

format_vulnerabilities({error, ErrorContext}) ->
    {error, ErrorContext};
format_vulnerabilities(ExistingVulnerabilities) when is_list(ExistingVulnerabilities) ->
    lists:map(fun ({{N, _}, Ids}) ->
                      io_lib:format("- ~s: ~s~n", [N, lists:join(",", Ids)])
              end, ExistingVulnerabilities).

report_vulnerabilities([]) ->
    io:format("[OSV] No new vulnerabilities reported.~n");
report_vulnerabilities({error, [URI, Error]}) ->
    fail("[OSV] POST request to ~p errors: ~p", [URI, Error]);
report_vulnerabilities(FormatVulns) ->
    io:format("[OSV] There are existing vulnerabilities:~n~s", [FormatVulns]).

osv_names(#{~"queries" := Packages}) ->
    lists:map(fun osv_names/1, Packages);
osv_names(#{~"package" := #{~"name" := Name }, ~"commit" := Commit}) ->
    {Name, Commit};
osv_names(#{~"package" := #{~"name" := Name }, ~"version" := Version}) ->
    {Name, Version}.


generate_osv_query(Packages) ->
    #{~"queries" => lists:usort(lists:foldl(fun generate_osv_query/2, [], Packages))}.
generate_osv_query(#{~"versionInfo" := Vsn, ~"ecosystem" := Ecosystem, ~"name" := Name}, Acc) ->
    Package = #{~"package" => #{~"name" => Name, ~"ecosystem" => Ecosystem}, ~"version" => Vsn},
    [Package | Acc];
generate_osv_query(#{~"sha" := SHA, ~"downloadLocation" := Location}, Acc) ->
    case string:prefix(Location, ~"https://") of
        nomatch ->
            Acc;
        URI ->
            Package = #{~"package" => #{~"name" => URI}, ~"commit" => SHA},
            [Package | Acc]
    end;
generate_osv_query(_, Acc) ->
    Acc.

%% when we no longer need to maintain maint-27, we can remove
%% this hard-coded commits and versions.
vendor_by_version(~"maint-26") ->
    #{~"queries" =>
          [#{%% v1.2.13
             ~"commit"=> ~"04f42ceca40f73e2978b50e93806c2a18c1281fc",
             ~"package"=> #{~"name"=> ~"github.com/madler/zlib"}},

           #{~"commit"=> ~"915186f6c5c2f5a4638e5cb97ccc23d741521a64",
             ~"package"=> #{~"name"=> ~"github.com/asmjit/asmjit"}},

           #{~"commit"=> ~"e745bad3b1d05b5b19ec652d68abb37865ffa454",
             ~"package"=> #{~"name"=> ~"github.com/microsoft/STL"}},

           #{~"commit"=> ~"844864ac213bdbf1fb57e6f51c653b3d90af0937",
             ~"package"=> #{~"name"=> ~"github.com/ulfjack/ryu"}},

           #{% 3.1.4
             ~"commit"=> ~"01d5e2318405362b4de5e670c90d9b40a351d053",
             ~"package"=> #{~"name"=> ~"github.com/openssl/openssl"}},

           #{% 8.45, not offial but the official sourceforge is not available
             ~"commit"=> ~"3934406b50b8c2a4e2fc7362ed8026224ac90828",
             ~"package"=> #{~"name"=> ~"github.com/nektro/pcre-8.45"}},

           #{~"version"=> ~"2.32",
             ~"package"=> #{~"ecosystem"=> ~"npm",
                            ~"name"=> ~"tablesorter"}},

           #{~"version"=> ~"3.7.1",
             ~"package"=> #{~"ecosystem"=> ~"npm",
                            ~"name"=> ~"jquery"}}
          ]};
vendor_by_version(~"maint-27") ->
    #{~"queries" =>
          [#{ %% v1.2.13
             ~"commit"=> ~"04f42ceca40f73e2978b50e93806c2a18c1281fc",
             ~"package"=> #{~"name"=> ~"github.com/madler/zlib"}},

           #{~"commit"=> ~"a465fe71ab3d0e224b2b4bd0fac69ae68ab9239d",
             ~"package"=> #{ ~"name"=> ~"github.com/asmjit/asmjit"}},

           #{~"commit"=> ~"e745bad3b1d05b5b19ec652d68abb37865ffa454",
             ~"package"=> #{~"name"=> ~"github.com/microsoft/STL"}},

           #{~"commit"=> ~"844864ac213bdbf1fb57e6f51c653b3d90af0937",
             ~"package"=>#{~"name"=> ~"github.com/ulfjack/ryu"}},

           #{  % 3.1.4
             ~"commit"=> ~"01d5e2318405362b4de5e670c90d9b40a351d053",
             ~"package"=> #{~"name"=> ~"github.com/openssl/openssl"}},

           #{% 8.45, not offial but the official sourceforge is not available
             ~"commit"=> ~"3934406b50b8c2a4e2fc7362ed8026224ac90828",
             ~"package"=> #{ ~"name"=> ~"github.com/nektro/pcre-8.45"}},

           #{~"version"=> ~"2.32",
             ~"package"=> #{~"ecosystem"=> ~"npm",
                            ~"name"=> ~"tablesorter"}},

           #{~"version"=> ~"3.7.1",
             ~"package"=> #{~"ecosystem"=> ~"npm",
                            ~"name"=> ~"jquery"}}
          ]};
vendor_by_version(_) ->
    VendorSrcFiles = find_vendor_src_files("."),
    Packages = generate_vendor_info_package(VendorSrcFiles),
    Packages1 = ignore_non_vulnerable_vendors(Packages),
    generate_osv_query(Packages1).

%% OTP only vendors the documentation from wx, so we can ignore
%% any vulnerability. The user should still look into possible
%% issues with wx if they link to it.
non_vulnerable_vendor_packages() ->
    [~"wx-doc-src"].

ignore_non_vulnerable_vendors(Packages) ->
    lists:filter(fun (#{~"ID" := Id}) -> not lists:member(Id, non_vulnerable_vendor_packages())
                 end, Packages).

cleanup_path(<<"./", Path/binary>>) when is_binary(Path) -> Path;
cleanup_path(Path) when is_binary(Path) -> Path.

build_package_location(<<>>) -> #{};
build_package_location(AppSrcPath) ->
    case string:split(AppSrcPath, "/", all) of
        [~"lib", App | _] ->
            AppName = erlang:binary_to_atom(App),
            _ = case application:load(AppName) of
                    R when R==ok orelse R=={error, {already_loaded, AppName}} ->
                        %% somewhat unsafe binary_to_atom/1 but we have guarantees to receive
                        %% only apps in Erlang/OTP
                        {ok, AppKey} = application:get_all_key(AppName),
                        AppKey1 = app_key_to_record(AppKey),
                        #{App => {<<"lib/", App/binary>>, AppKey1}};
                    _E ->
                        % useful only for debugging.
                        % this script should have all dependencies and never end up here.
                        io:format("[Error] ~p~n", [{AppSrcPath, _E, AppName, App}]),
                        error(?FUNCTION_NAME)
                end;
        [~"erts"=Erts | _] ->
            #{Erts => {Erts, #app_info{ description = ~"Erlang Runtime System",
                                        id           = [],
                                        vsn          = erlang:list_to_binary(erlang:system_info(version)),
                                        modules      = not_loaded,
                                        applications = [],
                                        included_applications = [],
                                        optional_applications = [] }}}
    end.

app_key_to_record(AppKey) ->
    [{description, Description}, {id, Id},
     {vsn, Vsn}, {modules, Modules},
     {maxP, _}, {maxT, _},
     {registered, _Registered},
     {included_applications, Included},
     {optional_applications, Optional},
     {applications, Apps},
     {env, _Env}, {mod, _Mod},
     {start_phases,_Phases}] = AppKey,
    #app_info{ description  = erlang:list_to_binary(Description),
               id           = erlang:list_to_binary(Id),
               vsn          = erlang:list_to_binary(Vsn),
               modules      = Modules,
               applications = Apps,
               included_applications = Included,
               optional_applications = Optional }.


-spec generate_spdx_packages(PackageMappings, Spdx) -> [spdx_package()] when
      PackageMappings :: #{AppName => {AppPath, app_info()}},
      AppName         :: unicode:chardata(),
      AppPath         :: unicode:chardata(),
      Spdx            :: map().
generate_spdx_packages(PackageMappings, #{~"files" := Files,
                                          ~"documentDescribes" := [ProjectName]}=_Spdx) ->
    SystemDocs = generate_spdx_system_docs(Files, ProjectName),
    maps:fold(fun (PackageName, {PrefixPath, AppInfo}, Acc) ->
                      SpdxPackageFiles = group_files_by_app(Files, PrefixPath),
                      TestFiles = get_test_files(PackageName, SpdxPackageFiles, PrefixPath),
                      DocFiles = get_doc_files(PackageName, SpdxPackageFiles, PrefixPath),
                      OTPAppFiles = (SpdxPackageFiles -- TestFiles) -- DocFiles,

                      LicenseOTPApp = otp_app_license_mapping(PackageName),
                      Package = create_spdx_package_record(PackageName, AppInfo#app_info.vsn,
                                                           AppInfo#app_info.description,
                                                           OTPAppFiles, ?spdx_homepage,
                                                           LicenseOTPApp,LicenseOTPApp, true),
                      DocPackage = create_spdx_package_record(<<PackageName/binary, "-documentation">>,
                                                              AppInfo#app_info.vsn,
                                                              <<"Documentation of ", PackageName/binary>>,
                                                              DocFiles, ?spdx_homepage,
                                                              LicenseOTPApp, LicenseOTPApp, false),
                      TestPackage = create_spdx_package_record(<<PackageName/binary, "-test">>,
                                                              AppInfo#app_info.vsn,
                                                              <<"Tests of ", PackageName/binary>>,
                                                              TestFiles, ?spdx_homepage,
                                                              LicenseOTPApp, LicenseOTPApp, false),

                      Relations = [ {'PACKAGE_OF', [{ Package#spdx_package.'SPDXID', ProjectName }]},
                                    {'DOCUMENTATION_OF', [{ DocPackage#spdx_package.'SPDXID', Package#spdx_package.'SPDXID' }]},
                                    {'TEST_OF', [{ TestPackage#spdx_package.'SPDXID', Package#spdx_package.'SPDXID' }]} ],

                      Packages = lists:zipwith(fun (P, {K, R}) ->
                                                       P#spdx_package { 'relationships' = #{ K => R} }
                                               end, [Package, DocPackage, TestPackage], Relations),
                      Packages ++ Acc
               end, [SystemDocs], PackageMappings).

generate_spdx_system_docs(Files, ParentSPDXPackageId) ->
    PrefixPath = ~"system",
    SpdxPackageFiles = group_files_by_app(Files, PrefixPath),
    PackageName = ~"system",
    DocFiles = get_doc_files(PackageName, SpdxPackageFiles, PrefixPath),
    LicenseUpdated = generate_license_info_from_files(DocFiles),
    ValidLicense = remove_invalid_spdx_licenses(LicenseUpdated),
    OneLinerLicense = binary:join(ValidLicense, ~" AND "),
    DocPackage = create_spdx_package_record(<<PackageName/binary, "-documentation">>,
                                            get_otp_version(),
                                            <<"System Documentation">>,
                                            DocFiles, ?spdx_homepage,
                                            OneLinerLicense, OneLinerLicense, false),
    Relations = #{ 'DOCUMENTATION_OF' => [{ DocPackage#spdx_package.'SPDXID', ParentSPDXPackageId }]},
    DocPackage#spdx_package { 'relationships' = Relations }.

%% Erlang/OTP apps always follow the convention of having 'test' and 'doc'
%% folder at top-level of the app folder. erts is more special and we must check
%% that in multiples levels, thus, we use wildcard patterns.
get_test_files(~"erts", SpdxPackageFiles, PrefixPath) ->
    group_files_by_folder(SpdxPackageFiles, binary_to_list(PrefixPath)++"/**/test/**");
get_test_files(_App, SpdxPackageFiles, PrefixPath) ->
    group_files_by_folder(SpdxPackageFiles, binary_to_list(PrefixPath)++"/test/**").

get_doc_files(~"erts", SpdxPackageFiles, PrefixPath) ->
    group_files_by_folder(SpdxPackageFiles, binary_to_list(PrefixPath)++"/**/doc/**");
get_doc_files(_App, SpdxPackageFiles, PrefixPath) ->
    group_files_by_folder(SpdxPackageFiles, binary_to_list(PrefixPath)++"/doc/**").

create_spdx_package_record(PackageName, Vsn, Description, SpdxPackageFiles,
                           Homepage, LicenseConcluded, LicenseDeclared, Purl) ->
    SpdxPackageName = generate_spdxid_name(PackageName),
    VerificationCodeValue = generate_verification_code_value(SpdxPackageFiles),
    Purl1 = case Purl of
                false -> false;
                true -> [create_externalRef_purl(Description, otp_purl(PackageName, Vsn)),
                         fix_openvex_reference()]
            end,
    #spdx_package {
       'SPDXID' = SpdxPackageName,
       'versionInfo' = Vsn,
       'description' = Description,
       'name' = PackageName,
       'copyrightText' = generate_copyright_text(SpdxPackageFiles),
       'filesAnalyzed' = true,

       %% O(n2) complexity... fix if necessary
       'hasFiles' = generate_has_files(SpdxPackageFiles),

       'purl' = Purl1,
       'homepage' = Homepage,
       'licenseConcluded' = LicenseConcluded,
       'licenseDeclared'  = LicenseDeclared,
       'licenseInfoFromFiles' = generate_license_info_from_files(SpdxPackageFiles),
       'packageVerificationCode' = #{ 'packageVerificationCodeValue' => VerificationCodeValue},
       'relationships' = #{}
      }.


fix_openvex_reference() ->
    OTPMajorVersion = hd(string:split(get_otp_version(), ".")),
    Reference = openvex_iri(OTPMajorVersion),
    #{
     ~"referenceCategory" => ~"SECURITY",
     ~"referenceLocator" => Reference,
     ~"referenceType" => ~"advisory"
    }.

%% Branch = ~"28" or similar. just the current version number.
openvex_iri(Branch) when is_binary(Branch) ->
    <<"https://erlang.org/download/vex/otp-", Branch/binary, ".openvex.json">>.

otp_app_license_mapping(Name) ->
    case Name of
        ~"edoc" -> ~"Apache-2.0 OR LGPL-2.1-or-later";
        ~"syntax_tools" -> ~"Apache-2.0 OR LGPL-2.1-or-later";
        ~"eunit" -> ~"Apache-2.0 OR LGPL-2.1-or-later";
        ~"eldap" -> ~"MIT";
        _ -> ?erlang_license
    end.


generate_spdxid_name(PackageName) ->
    PackageName1 = generate_spdx_valid_name(PackageName),
    <<"SPDXRef-otp-", PackageName1/binary>>.

undo_spdxid_name(Name) ->
    <<"SPDXRef-otp-", PackageName/binary>> = Name,
    PackageName.

generate_spdx_valid_name(PackageName) ->
    iolist_to_binary(string:replace(PackageName, ~"_", ~"", all)).

generate_license_info_from_files(SpdxPackageFiles) ->
    Result = lists:foldl(fun (#{~"licenseInfoInFiles" := LicenseInfoInFiles}, AccLicenses) ->
                                 split_licenses_in_individual_parts(LicenseInfoInFiles) ++ AccLicenses
                         end, [], SpdxPackageFiles),
    lists:uniq(Result).

split_licenses_in_individual_parts(Licenses) ->
    lists:foldl(fun (License, Acc) ->
                        L = re:replace(License, "[\(|\)]", "", [global, {return, list}]),
                        Licenses0 = string:split(list_to_binary(L), ~" OR "),
                        Licenses1 = lists:foldl(fun (L1, Acc1) -> string:split(L1, ~" AND ") ++ Acc1  end, [], Licenses0),
                        lists:uniq(lists:map(fun string:trim/1, Licenses1 ++ Acc))
                end, [], Licenses).

generate_has_files(SpdxPackageFiles) ->
    lists:map(fun (#{~"SPDXID" := SpdxId}) -> SpdxId end, SpdxPackageFiles).

%% alg. described in https://spdx.github.io/spdx-spec/v2.2.2/package-information/#791-description
-spec generate_verification_code_value(SpdxPackageFiles :: [SPDXFile :: map()]) -> binary().
generate_verification_code_value(SpdxPackageFiles) ->
    SHA1s = lists:map(fun (#{~"checksums" := [#{~"algorithm" := ~"SHA1", ~"checksumValue" := SHA1}]}) ->
                              SHA1
                      end, SpdxPackageFiles),
    Sorted = lists:sort(SHA1s),
    Merged = lists:foldl(fun(SHA1, Acc) ->
                                 <<Acc/binary, SHA1/binary>>
                         end, <<>>, Sorted),

    %% The crypto hash returns a binary that's not HEX encoded.
    HEX = binary:encode_hex(crypto:hash(sha, Merged)),

    %% encode_hex returns uppercase letters, but the output to SPDX must be lowercase
    StringHex = binary_to_list(HEX),
    list_to_binary(string:to_lower(StringHex)).

generate_copyright_text(SpdxPackageFiles) ->
    CopyrightText = lists:foldl(fun (#{~"copyrightText" := CopyrightText}, Acc0) ->
                                        lists:uniq([CopyrightText | Acc0])
                                end, [], SpdxPackageFiles),
    lists:foldl(fun (Copyright, Acc0) ->
                    <<Copyright/binary, "\n", Acc0/binary>>
                end, <<>>, CopyrightText).

group_files_by_app(Files, PrefixPath) ->
    lists:filter(fun (#{~"fileName" := Filename}) ->
                         case string:prefix(Filename, PrefixPath) of
                             nomatch ->
                                 false;
                             _ ->
                                 true
                         end
                 end, Files).

group_files_by_folder(Files, Wildcard) ->
    FilesInFolder = lists:map(fun unicode:characters_to_binary/1, filelib:wildcard(Wildcard)),
    lists:filter(fun (#{~"fileName" := Filename}) ->
                         lists:member(Filename, FilesInFolder)
                 end, Files).

test_file(#{sbom_file := SbomFile, ntia_checker := Verification}) ->
    Sbom = decode(SbomFile),
    ok = test_generator(Sbom),
    ok = test_ntia_checker(Verification, SbomFile),
    ok.

test_ntia_checker(false, _SbomFile) -> ok;
test_ntia_checker(true, SbomFile) ->
    have_tool("ntia-checker"),
    Cmd = "sbomcheck --comply ntia --file " ++ SbomFile,
    io:format("~nRunning: NTIA Compliance Checker~n[~ts]~n", [Cmd]),
    _ = cmd(Cmd),
    io:format("OK~n"),
    ok.

cmd(Cmd) ->
    string:trim(os:cmd(unicode:characters_to_list(Cmd),
                       #{ exception_on_failure => true })).

have_tool(Tool) ->
    case os:find_executable(Tool) of
        false -> fail("Could not find '~ts' in PATH", [Tool]);
        _ -> ok
    end.

fail(Fmt, Args) ->
    io:format(standard_error, Fmt++"\n", Args),
    erlang:halt(1).

test_generator(Sbom) ->
    io:format("~nRunning: verification of OTP SBOM integrity~n"),
    ok = project_generator(Sbom),
    ok = package_generator(Sbom),
    ok.

-define(CALL_TEST_FUNCTIONS(Tests, Sbom),
         (begin
            io:format("[~s]~n", [?FUNCTION_NAME]),
            lists:all(fun (Fun) ->
                              Module = ?MODULE,
                              Result = apply(Module, Fun, [Sbom]),
                              L = length(atom_to_list(Fun)),
                              io:format("- ~s~s~s~n", [Fun, lists:duplicate(40 - L, "."), Result]),
                              ok == Result
                      end, Tests)
        end)).

project_generator(Sbom) ->
    Tests = [test_project_name,
             test_name,
             test_creators_tooling,
             test_spdx_version],
    true = ?CALL_TEST_FUNCTIONS(Tests, Sbom),
    ok.

package_generator(Sbom) ->
    Tests = [test_minimum_apps,
             test_copyright_not_empty,

             %% TODO: enable once we can curate ORT copyrights
             %% test_copyright_format,

             test_filesAnalised,
             test_hasFiles_not_empty,

             % TODO: enable once licenseInFiles match licenseConcluded
             %% test_files_licenses,
             test_homepage,
             test_licenseConcluded_exists,
             test_licenseDeclared_exists,
             test_licenseInfoFromFiles_not_empty,
             test_package_names,
             test_package_ids,
             test_erts,
             test_verificationCode,
             test_supplier_Ericsson,
             test_originator_Ericsson,
             test_versionInfo_not_empty,
             test_package_hasFiles,
             test_project_purl,
             test_packages_purl,
             test_download_location,
             test_download_vendor_location,
             test_package_relations,
             test_has_extracted_licenses,
             test_vendor_packages],
    true = ?CALL_TEST_FUNCTIONS(Tests, Sbom),
    ok.

test_project_name(#{~"documentDescribes" := [ProjectName]}=_Sbom) ->
    ?spdxref_project_name = ProjectName,
    ok.

test_name(#{~"name" := Name}=_Sbom) ->
    ?spdx_project_name = Name,
    ok.

test_creators_tooling(#{~"creationInfo" := #{~"creators" := Creators}}=_Sbom) ->
    true = lists:any(fun (Name) ->
                             case string:prefix(Name, ?spdx_creators_tooling) of
                                 nomatch -> false;
                                 _ -> true
                             end
                     end, Creators),
    ok.

test_spdx_version(#{~"spdxVersion" := Version}=_Sbom) ->
    ?spdx_version = Version,
    ok.

test_minimum_apps(#{~"documentDescribes" := [ProjectName], ~"packages" := Packages}=_Sbom) ->
    _ = lists:foreach(fun (X) -> application:load(erlang:binary_to_atom(X)) end, minimum_otp_apps()),
    [#{~"name" := Project}] = lists:filter(fun (#{~"SPDXID" := Id}) -> Id == ProjectName end, Packages),
    TestPackageNames = [Project | minimum_otp_apps() ++ root_vendor_packages()],
    SPDXIds = lists:map(fun (#{~"name" := SPDXId}) -> SPDXId end, Packages),
    try
        %% test know packages are captured
        true = [] == TestPackageNames -- SPDXIds
    catch
        _E:_S:_ ->
            io:format("Minimum apps not captured.~n~p distinct from ~p~n", [TestPackageNames -- SPDXIds, SPDXIds -- TestPackageNames]),
            error(?FUNCTION_NAME)
    end,
    AppNamesVersion = lists:map(fun ({Name, Version}) -> {generate_spdxid_name(Name), Version} end, get_otp_apps_from_table()),
    true = lists:all(fun (#{~"SPDXID" := Id, ~"versionInfo" := Version}) ->
                              case lists:keyfind(Id, 1, AppNamesVersion) of
                                  {_, TableVersion} ->
                                      TableVersion == Version;
                                  false ->
                                      true
                              end
                      end, Packages),
    ok.

minimum_otp_apps() ->
    [~"kernel", ~"stdlib", ~"xmerl", ~"wx", ~"tools", ~"tftp", ~"syntax_tools", ~"ssl",
     ~"ssh", ~"snmp", ~"sasl", ~"runtime_tools", ~"reltool", ~"public_key", ~"parsetools",
     ~"os_mon", ~"observer", ~"mnesia", ~"megaco", ~"jinterface", ~"inets", ~"ftp", ~"eunit",
     ~"et", ~"erl_interface", ~"eldap", ~"edoc", ~"diameter", ~"dialyzer", ~"debugger", ~"crypto",
     ~"compiler", ~"common_test", ~"erts", ~"asn1", ~"odbc"].

root_vendor_packages() ->
    [ ~"asmjit", ~"pcre2", ~"zlib", ~"ryu", ~"zstd"].

minimum_vendor_packages() ->
    %% self-contained
    root_vendor_packages() ++
        [~"tcl", ~"STL", ~"json-test-suite", ~"openssl", ~"Autoconf", ~"wx-doc-src", ~"jquery", ~"tablesorter"].

test_copyright_not_empty(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"copyrightText" := Copyright}) -> Copyright =/= ~"" end, Packages),
    ok.

%% test_copyright_format(#{~"packages" := Packages, ~"files" := Files}) ->
%%     EricssonRegex = ~S"^Copyright Ericsson AB ((?:19|20)[0-9]{2}-)?((?:19|20)[0-9]{2}).*$",
%%     ContributorRegex = ~S"^Copyright([\s]?\([cC©]\))? ((?:19|20)[0-9]{2}-)?((?:19|20)[0-9]{2}) ((\w|\s|-)*)<(\w|\.|-)+@(\w|\.|-)+>$",
%%     VendorRegex = ~S"^Copyright([\s]?\([cC©]\))? ((?:19|20)[0-9]{2}-)?((?:19|20)[0-9]{2})?((\w|\s|-|,|\.)*)$",
%%     Default = ~S"^Copyright[\s]?(\([cC©]\))? ((?:19|20)[0-9]{2}-)?((?:19|20)[0-9]{2}) Erlang/OTP and its contributors$",
%%     NoAssertionRegex = "^NOASSERTION|NONE",
%%     Regexes = [EricssonRegex, ContributorRegex, VendorRegex, NoAssertionRegex, Default],

%%     Regex = lists:concat(lists:join(~S"|", Regexes)),
%%     {ok, CopyrightRegex} = re:compile([Regex]),
%%     true = lists:all(fun (#{~"copyrightText" := CopyrightText, ~"fileName" := Filename}) ->
%%                              Copyrights = string:split(CopyrightText, "\n", all),
%%                              lists:all(fun (C) ->
%%                                                case re:run(C, CopyrightRegex) of
%%                                                    nomatch ->
%%                                                        throw({warn, "Invalid Copyright: '~ts' in '~ts for ~ts~n'", [C, Filename, Regex]});
%%                                                    _ ->
%%                                                        true
%%                                                end
%%                                        end, Copyrights)
%%                      end, Files),

%%     true = lists:all(fun (#{~"copyrightText" := CopyrightText}) ->
%%                              Copyrights = string:split(CopyrightText, "\n", all),
%%                              lists:all(fun (C) ->
%%                                                case re:run(C, CopyrightRegex) of
%%                                                    nomatch ->
%%                                                        throw({warn, "Invalid Copyright: '~ts'", [C]});
%%                                                    _ ->
%%                                                        true
%%                                                end
%%                                        end, Copyrights)
%%                      end, Packages),
%%     ok.


test_filesAnalised(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"filesAnalyzed" := Bool}) -> Bool = true end, Packages),
    ok.

test_hasFiles_not_empty(#{~"packages" := Packages}) ->
    try
        true = lists:all(fun (#{~"hasFiles" := Files}) -> length(Files) > 0 end, Packages)
    catch
        _:_:_ ->
            lists:foreach(fun (#{~"hasFiles" := Files, ~"SPDXID":=Id}) ->
                              io:format("~p: length: ~p~n", [Id, length(Files)])
                      end, Packages),
            error(?FUNCTION_NAME)
    end,
    ok.

%% test_files_licenses(Input) ->
%%     ok = test_concluded_license_equals_license_in_file(Input),
%%     ok.

%% print_error(false, Input) ->
%%     io:format("[~p] ~p~n", [false, Input]),
%%     false;
%% print_error(true, _Input) ->
%%     true.

%% test_concluded_license_equals_license_in_file(#{~"files" := Files}) ->
%%     true = lists:all(fun (#{~"licenseInfoInFiles" := [License], ~"licenseConcluded" := License}) ->
%%                              true;
%%                          (#{~"licenseInfoInFiles" := [~"NONE"]}) ->
%%                              true;
%%                          (#{~"licenseInfoInFiles" := Licenses,
%%                             ~"licenseConcluded" := Concluded,
%%                             ~"SPDXID" := Id}) when length(Licenses) > 1 ->
%%                              Licenses1 = lists:map(fun erlang:binary_to_list/1, Licenses),
%%                              LicensesBin = erlang:list_to_binary(lists:join(" AND ", Licenses1)),
%%                              print_error(Concluded =:= LicensesBin, {Id, Licenses, Concluded, ?LINE});
%%                          (#{~"licenseInfoInFiles" := Licenses,
%%                             ~"licenseConcluded" := Concluded,
%%                             ~"SPDXID" := Id}) ->
%%                              print_error(Concluded =:= Licenses, {Id, Licenses, Concluded, ?LINE})
%%                      end, Files),
%%     ok.

test_homepage(#{~"packages" := Packages})->
    true = lists:all(fun (#{~"homepage" := Homepage}) -> Homepage == ?spdx_homepage orelse Homepage =/= <<>> end, Packages),
    ok.

test_licenseConcluded_exists(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"licenseConcluded" := License}) -> License =/= ~"" andalso License =/= ~"NONE" end, Packages),
    ok.

test_licenseDeclared_exists(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"licenseDeclared" := License}) -> License =/= ~"" andalso License =/= ~"NONE" end, Packages),
    ok.

test_licenseInfoFromFiles_not_empty(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"licenseInfoFromFiles" := Ls}) ->
                             case Ls of
                                 [] ->
                                     false;
                                 [L | _] when is_list(L) ->
                                     false;
                                 _ ->
                                     true = lists:all(fun (License) -> not erlang:is_integer(License) end, Ls)
                             end
                     end, Packages),

    %% check no duplicates
    true = lists:all(fun (#{~"licenseInfoFromFiles" := Ls}) ->
                             erlang:length(lists:uniq(Ls)) == erlang:length(Ls)
                     end, Packages),
    ok.

test_package_names(#{~"packages" := Packages}) ->
    %% not repeated names
    Names = lists:map(fun (#{~"name" := Name}) -> Name end, Packages),

    %% we know openssl is repeated twice, and Autconf is placed in multiple packages.
    SkippedNames = [~"openssl", ~"Autoconf"],
    Names1 = lists:filter(fun (N) -> not lists:member(N, SkippedNames) end, Names),
    try
        true = length(Names1) == length(lists:uniq(Names1))
    catch
        _:_:_ ->
            io:format("Names are not unique: ~p -- ~p", [Names1, lists:uniq(Names1)]),
            error(?FUNCTION_NAME)
    end,

    true = lists:all(fun (N) -> lists:member(N, Names) end, minimum_otp_apps()),
    ok.

test_package_ids(#{~"packages" := Packages}) ->
    %% Test name starts with SPDXRef-, and contains alphanumeric and -
    true = lists:all(fun (#{~"SPDXID" := <<"SPDXRef-", Rest/binary>>}) ->
                             %% Match on alphanumeric and -
                             Query = "^[a-zA-Z0-9-]*$",
                             {match, _} = re:run(Rest, Query),
                             true
                     end, Packages),
    ok.

test_erts(#{~"packages" := Packages, ~"files" := Files}) ->
    ErtsSpdxId = generate_spdxid_name(~"erts"),
    ErtsPkg = lists:search(fun (#{~"SPDXID" := SpdxId}) -> SpdxId == ErtsSpdxId end, Packages),
    {value, #{~"hasFiles" := HasFiles}} = ErtsPkg,

    %% checks that there are no test files in erts package.
    %% test files for erts should be in erts-test
    ErtsTestFiles = lists:filtermap(fun (#{~"fileName" := <<"erts/emulator/test/", _/binary>>,
                                           ~"SPDXID" := FileId}) -> {true, FileId};
                                        (#{~"fileName" := <<"erts/test/", _/binary>>,
                                           ~"SPDXID" := FileId}) -> {true, FileId};
                                        (_) -> false
                                    end, Files),
    HasFiles = HasFiles -- ErtsTestFiles,

    %% checks that there are no doc files in erts package.
    %% doc files for erts should be in erts-doc
    ErtsDocFiles = lists:filtermap(fun (#{~"fileName" := <<"erts/preloaded/doc/", _/binary>>,
                                          ~"SPDXID" := FileId}) -> {true, FileId};
                                       (#{~"fileName" := <<"erts/doc/", _/binary>>,
                                          ~"SPDXID" := FileId}) -> {true, FileId};
                                       (_) -> false
                                   end, Files),
    HasFiles = HasFiles -- ErtsDocFiles,
    ok.

test_verificationCode(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"packageVerificationCode" := #{~"packageVerificationCodeValue" := Value}}) ->
                             Value =/= ~"TODO" andalso Value =/= <<>>
                     end, Packages),
    ok.

test_supplier_Ericsson(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"supplier" := Supplier, ~"name" := Name}) ->
                             %% logical implication (->) expressed in boolean logic (not A or B)
                             not lists:member(Name, minimum_otp_apps()) orelse Supplier == ?spdx_supplier
                     end, Packages),
    ok.

test_originator_Ericsson(#{~"packages" := Packages}) ->
    %% TODO: needs fixing ORT otp
    true = lists:all(fun (#{~"name" := Name}=Spdx) ->
                             case maps:get(~"originator", Spdx, badkey) of
                                 badkey ->
                                     true;
                                 Originator ->
                                     %% logical implication (->) expressed in boolean logic (not A or B)
                                     not lists:member(Name, minimum_otp_apps()) orelse Originator == ?spdx_supplier
                             end
                     end, Packages),
    ok.

test_versionInfo_not_empty(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"versionInfo" := Version}) -> Version =/= ~"" end, Packages),
    ok.

test_download_location(#{~"packages" := Packages}) ->
    true = lists:all(fun (#{~"downloadLocation" := Loc}) -> Loc =/= ~"" end, Packages),
    ok.

%% vendor location should use https://github.com where possible due to integration with OSV.
%% see generate_osv_query/1.
test_download_vendor_location(#{~"packages" := Packages}) ->
    %% update list below if new runtime dependencies without git repo appear.
    KnownExcludedNames = [~"Autoconf", ~"tcl", ~"Unicode Character Database"],
    true = lists:all(fun (#{~"downloadLocation" := Loc, ~"name" := Name}) ->
                             lists:member(Name, KnownExcludedNames)
                                 orelse string:prefix(Loc, ~"https://github.com") =/= nomatch
                     end, Packages),
    ok.

test_package_hasFiles(#{~"packages" := Packages}) ->
    %% test files are not repeated
    AllFiles = lists:foldl(fun (#{~"hasFiles" := FileIds}, Acc) -> FileIds ++ Acc end, [], Packages),

    try
        true = length(AllFiles) == length(lists:uniq(AllFiles))
    catch _:_:_ ->
            io:format("~p~n",[AllFiles -- lists:uniq(AllFiles)]),
            error(?FUNCTION_NAME)
    end,

    %% Test all files contain at least one file
    true = lists:all(fun (#{~"hasFiles" := Files}) -> erlang:length(Files) > 0 end, Packages),
    ok.

test_project_purl(#{~"documentDescribes" := [ProjectName], ~"packages" := Packages}=_Sbom) ->
    [#{~"externalRefs" := [Purl], ~"versionInfo" := VersionInfo}] = lists:filter(fun (#{~"SPDXID" := Id}) -> ProjectName == Id end, Packages),
    RefLoc = ?spdx_project_purl,
    true = Purl == RefLoc#{ ~"referenceLocator" := <<?ErlangPURL, "@OTP-", VersionInfo/binary>> },
    ok.

test_packages_purl(#{~"documentDescribes" := [ProjectName], ~"packages" := Packages}=_Sbom) ->
    OTPPackages = lists:filter(fun (#{~"SPDXID" := Id, ~"name" := Name}) ->
                                       ProjectName =/= Id andalso lists:member(Name, minimum_otp_apps())
                               end, Packages),
    true = lists:all(fun (#{~"name" := Name, ~"versionInfo" := Version,
                            ~"externalRefs" := [#{~"referenceLocator":= RefLoc}=Ref,
                                                OpenVex]}) ->
                             ExternalRef = create_externalRef_purl(~"", otp_purl(Name, Version)),
                             ExternalRef1 = maps:remove(~"comment", ExternalRef),
                             Ref1 = maps:remove(~"comment", Ref),

                             ExpectedVEX = fix_openvex_reference(),

                             %% check expected external ref
                             ExternalRef1 =:= Ref1  andalso
                                 %% check metadata is included in purl
                                 nomatch =/= string:find(RefLoc, ?spdx_purl_meta_data) andalso
                                 ExpectedVEX == OpenVex
                     end, OTPPackages),
    ok.

test_vendor_packages(Sbom) ->
    ok = minimum_vendor_packages(Sbom),
    ok = vendor_relations(Sbom),
    ok.

minimum_vendor_packages(#{~"packages" := Packages}=_Sbom) ->
    VendorNames = minimum_vendor_packages(),
    Names = lists:map(fun (#{~"name" := Name}) -> Name end, Packages),
    true = [] == VendorNames -- Names,
    ok.

vendor_relations(#{~"packages" := Packages, ~"relationships" := Relations}) ->
    PackageIds = lists:map(fun (#{~"SPDXID" := Id}) -> Id end, Packages),
    VendorIds = lists:filtermap(fun (#{~"comment" := " vendor package", ~"SPDXID" := Id}) -> {true, Id} ;
                                      (_) -> false
                                  end, Packages),
    true = lists:all(fun (#{~"relatedSpdxElement" := Related,
                            ~"relationshipType"   := _,
                            ~"spdxElementId" := PackageId}) ->
                             case lists:member(PackageId, VendorIds) of
                                 true ->
                                     lists:member(Related, PackageIds) andalso
                                         PackageId =/= Related ;
                                 false ->
                                     %% ignore non-vendor relations
                                     true
                             end
                     end, Relations),
    ok.

test_package_relations(#{~"packages" := Packages}=Spdx) ->
    PackageIds = lists:map(fun (#{~"SPDXID" := Id}) -> Id end, Packages),
    Relations = maps:get(~"relationships", Spdx),
    true = lists:all(fun (#{~"relatedSpdxElement" := Related,
                            ~"relationshipType"   := Relation,
                            ~"spdxElementId" := PackageId}=Rel) ->
                             Result =
                                 lists:member(Relation, [~"PACKAGE_OF", ~"DEPENDS_ON", ~"TEST_OF",
                                                         ~"OPTIONAL_DEPENDENCY_OF", ~"DOCUMENTATION_OF"]) andalso
                                 lists:member(Related, PackageIds) andalso
                                 lists:member(PackageId, PackageIds) andalso
                                 PackageId =/= Related andalso
                                 PackageId =/= ?spdxref_project_name,
                            case Result of
                                false ->
                                    io:format("Error in relation: ~p~n", [Rel]),
                                    false;
                                true ->
                                    true
                            end
                     end, Relations),

    %% test_known_special_cases(),
    SpecialCases = [#{~"relatedSpdxElement" => ~"SPDXRef-otp-erlinterface",
                      ~"relationshipType" => ~"PACKAGE_OF",
                      ~"spdxElementId" => ~"SPDXRef-otp-erlinterface-openssl"},
                    #{~"relatedSpdxElement" => ~"SPDXRef-otp-stdlib-test",
                      ~"relationshipType" => ~"PACKAGE_OF",
                      ~"spdxElementId" => ~"SPDXRef-otp-stdlib-test-json-suite"},
                    #{~"relatedSpdxElement" => ~"SPDXRef-otp-stdlib",
                      ~"relationshipType" => ~"PACKAGE_OF",
                      ~"spdxElementId" => ~"SPDXRef-otp-stdlib-unicode"},
                    #{~"relatedSpdxElement" => ~"SPDXRef-otp-commontest",
                      ~"relationshipType" => ~"PACKAGE_OF",
                      ~"spdxElementId" => ~"SPDXRef-otp-commontest-tablesorter"},
                    #{~"relatedSpdxElement" => ~"SPDXRef-otp-commontest",
                      ~"relationshipType" => ~"PACKAGE_OF",
                      ~"spdxElementId" => ~"SPDXRef-otp-commontest-jquery"}],
    true = lists:all(fun (Case) -> lists:member(Case, Relations) end, SpecialCases),
    ok.

test_has_extracted_licenses(#{~"hasExtractedLicensingInfos" := LicensesInfo,
                              ~"packages" := Packages}=_Spdx) ->
    LicenseRefsInProject =
        lists:uniq(
          lists:foldl(fun (#{~"licenseInfoFromFiles" := InfoFromFilesInPackage }, Acc) ->
                              LicenseRefs = lists:filter(fun (<<"LicenseRef-", _/binary>>) -> true ;
                                                             (_) -> false
                                                         end, InfoFromFilesInPackage),
                              LicenseRefs ++ Acc
                      end, [], Packages)),
    true = lists:all(fun (#{~"licenseId" := LicenseId}) -> lists:member(LicenseId, LicenseRefsInProject) end, LicensesInfo),
    ok.

%% Adds LicenseRef licenses where the text is missing.
extracted_license_info() ->
    [begin
         {ok, License} = file:read_file(Name),
         {unicode:characters_to_binary(filename:basename(filename:rootname(Name))), License}
     end || Name <- filelib:wildcard("LICENSES/LicenseRef*.txt")].

%%
%% REUSE-IgnoreEnd
%%

%% input: file points to the list of items openvex.table
%% branch: tell us which branch from openvex.table we take into account
%%
%% We take items from 'input.branch' and check that the openvex file
%% contains those exact changes. if not, a new change is issued
%%
%% Documentation in HOWTO/SBOM.md
%%

vex_path(Branch) ->
    VexPath = ?VexPath,
    vex_path(VexPath, Branch).
vex_path(VexPath, Branch) ->
    <<VexPath/binary, Branch/binary, ".openvex.json">>.

init_openvex(#{input_file := File, branch := Branch, vex_path := VexPath}) ->
    InitVex = vex_path(VexPath, Branch),
    VexStmts = case filelib:is_file(InitVex) of
                   true -> % file exists
                       maps:get(~"statements", decode(InitVex));
                   false -> % create file
                       Init = init_openvex_file(Branch),
                       file:write_file(InitVex, json:format(Init)),
                       maps:get(~"statements", Init)
               end,
    run_openvex1(VexStmts, File, Branch, VexPath).

run_openvex(#{input_file := File, branch := Branch, vex_path := VexPath}) ->
    InitVex = vex_path(VexPath, Branch),
    VexStmts = maps:get(~"statements", decode(InitVex)),
    run_openvex1(VexStmts, File, Branch, VexPath).

run_openvex1(VexStmts, VexTableFile, Branch, VexPath) ->
    Statements = calculate_statements(VexStmts, VexTableFile, Branch, VexPath),
    lists:foreach(fun (St) -> io:format("~ts", [St]) end, Statements).

verify_openvex(#{create_pr := PR}) ->
    Branches = get_supported_branches(),
    io:format("Sync ~p~n", [Branches]),
    _ = lists:foreach(
          fun (Branch) ->
                  case verify_openvex_advisories(Branch) of
                      [] ->
                          io:format("No new advisories nor OpenVEX statements created for '~s'.", [Branch]);
                      MissingAdvisories ->
                          io:format("Missing Advisories:~n~p~n~n", [MissingAdvisories]),
                          case PR of
                              false ->
                                  io:format("To automatically update openvex.table and create a PR run:~n" ++
                                                ".github/scripts/otp-compliance.es vex verify -b ~s -p~n~n", [Branch]);
                              true ->
                                  Advs = create_advisory(MissingAdvisories),
                                  _ = update_openvex_otp_table(Branch, Advs),
                                  BranchStr = erlang:binary_to_list(Branch),
                                  _ = cmd(".github/scripts/otp-compliance.es vex run -b "++ BranchStr ++ " | bash")
                          end
                  end
          end, Branches),
    case PR of
        true ->
            Result = cmd(".github/scripts/create-openvex-pr.sh " ++ ?GH_ACCOUNT ++ " vex"),
            io:format("~s~n", [unicode:characters_to_binary(Result)]);
        false ->
            ok
    end.

verify_openvex_advisories(Branch) ->
    OpenVEX = read_openvex_file(Branch),
    Advisory = download_advisory_from_branch(Branch),
    verify_advisory_against_openvex(OpenVEX, Advisory).

-spec get_supported_branches() -> [Branches :: binary()].
get_supported_branches() ->
    Branches = cmd(".github/scripts/get-supported-branches.sh"),
    BranchesBin = json:decode(erlang:list_to_binary(Branches)),
    io:format("~p~n~p~n", [Branches, BranchesBin]),
    lists:filtermap(fun (<<"maint-", _/binary>>=OTP) -> {true, maint_to_otp_conversion(OTP)};
                        (_) -> false
                 end, BranchesBin).

create_advisory(Advisories) ->
    lists:foldl(fun (Adv, Acc) ->
                        create_openvex_otp_entries(Adv) ++ Acc
                end, [], Advisories).

create_openvex_otp_entries(#{'CVE' := CVEId,
                             'appName' := AppName,
                             'affectedVersions' := AffectedVersions,
                             'fixedVersions' := FixedVersions}) ->
    AppFixedVersions = lists:map(fun (Ver) -> create_app_purl(AppName, Ver) end, FixedVersions),
    lists:map(fun (Affected) ->
                      Purl = create_app_purl(AppName, Affected),
                      create_openvex_app_entry(Purl, CVEId, AppFixedVersions)
              end, AffectedVersions).

create_app_purl(AppName, Version) when is_binary(AppName), is_binary(Version) ->
    <<"pkg:otp/", AppName/binary, "@", Version/binary>>.

create_openvex_app_entry(Purl, CVEId, FixedVersions) ->
    #{Purl => CVEId,
      ~"status" =>
          #{ ~"affected" => iolist_to_binary(io_lib:format("Update to any of the following versions: ~s", [FixedVersions])),
             ~"fixed" => FixedVersions}}.

update_openvex_otp_table(Branch, Advs) ->
    Path = ?OpenVEXTablePath,
    io:format("OpenVEX Statements:~n~p~n~n", [Advs]),
    #{Branch := Statements}=Table = decode(Path),
    UpdatedTable = Table#{Branch := Advs ++ Statements},
    io:format("Update table:~n~p~n", [UpdatedTable]),
    file:write_file(Path, json:format(UpdatedTable)).

generate_gh_link(Part) ->
    "\"/repos/erlang/otp/security-advisories?" ++ Part ++ "\"".

download_advisory_from_branch(Branch) ->
    Opts = ?GH_ADVISORIES_OPTIONS,
    Cmd = generate_gh_link(Opts),
    paginate_years(Branch, Cmd).

%%
%% Download GH Advisories for erlang/otp using
%% gh_advisories_options(). Download pages of information
%% until there are no more pages of advisories information
%% to download. Considers only information updated in the last
%% 5 years.
%%
paginate_years(Branch, Cmd) when is_binary(Cmd) ->
    paginate_years(Branch, erlang:binary_to_list(Cmd));
paginate_years(Branch, Cmd) when is_list(Cmd) ->
    Cmd0 = "gh api -i -H \"Accept: application/vnd.github+json\" -H \"X-GitHub-Api-Version: 2022-11-28\" ",
    Cmd1 = Cmd0 ++ Cmd,
    io:format("~p~n", [Cmd1]),
    AdvisoryStr = cmd(Cmd1),
    UnicodeBin = unicode:characters_to_binary(AdvisoryStr),
    RawHTTP = string:split(UnicodeBin, "\n", all),
    Body0 = extract_http_gh_body(RawHTTP),
    {{LowRangeYear, _Month, _Day}, _} = calendar:local_time(),

    %% Get the latest 5 years of CVEs. information is sorted
    case process_gh_page(LowRangeYear - ?GH_ADVISORIES_FROM_LAST_X_YEARS, Branch, Body0) of
        [] ->
            %% there was nothing useful based on the dates (sorted)
            %% so we do not need to continue pulling pages.
            [];
        [_|_]=Body1 ->
            %% there were CVE under the last ?GH_ADVISORIES_FROM_LAST_X_YEARS years.
            %% extract link to continue pulling GH pages
            case extract_http_gh_link(RawHTTP) of
                [NextQuery] ->
                    Body1 ++ paginate_years(Branch, NextQuery);
                [] ->
                    Body1
            end
    end.

process_gh_page(Year, Branch, Body) ->
    lists:foldl(fun (Vuln0, Acc0) ->
                        Vuln1 = filter_gh_cve_by({year, Year}, Vuln0),
                        [filter_gh_cve_by({otp, Branch}, Vuln1) | Acc0]
                end, [], Body).

filter_gh_cve_by({year, Year},
                 #{~"published_at" := <<Y1,Y2,Y3,Y4,_/binary>>}=Vuln) ->
    YYYY = erlang:binary_to_integer(<<Y1,Y2,Y3,Y4>>),
    case YYYY >= Year of
        true ->
            Vuln;
        false ->
            #{}
    end;
filter_gh_cve_by({otp, <<"otp-", Version/binary>>},
                 #{~"vulnerabilities" := Vulns}=CVE) ->
    %% Filters CVE based on version to scan.
    %% Example: {otp, ~"otp-27"} will filter out vulnerabilities from Vulns
    %% that affect OTP-25 applications. The algorithm updates the vulnerable_version_range
    %% to point to the most precise version of the vulnerability. In some cases, OTP refers
    %% to really old versions, so one must fetch the OTP-XX.0 version of the application
    %% in question.
    %%
    %% this algorithm is not general enough to be able to deal with all possible
    %% ways in which the CNA can report errors. and assumes the standard
    %% of having one vulnerable_version_range in the form <<">= 3.0">>.
    %%
    CVE#{ ~"vulnerabilities" :=
              lists:foldl(fun (#{~"package" := #{~"name" := ~"OTP"}}, Acc) ->
                                  %% ignore OTP release versions. We can generate these ones from
                                  %% the app specific version.
                                  Acc;
                              (#{~"package" := #{~"name" := AppName},
                                 ~"vulnerable_version_range" := VulnerableVersion,
                                 ~"patched_versions" := AppVersions}=Pkg, Acc) when is_binary(AppVersions) ->
                                  AppVersions1 =
                                      get_otp_app_version_from_gh_vulnerability(Version, VulnerableVersion, AppName, AppVersions),
                                  [Pkg#{~"patched_versions" := A,
                                        ~"vulnerable_version_range" := V} ||  {A, V} <- AppVersions1] ++ Acc
                          end, [], Vulns)}.

%% Input: <<"27">> and <<">= 3.2">> and <<"ssl">>, and <<"4.15.3, 5.1.5, 5.2.9">>
%% Output: [{~"27.3.3", ~"4.15.3"}]
-spec get_otp_app_version_from_gh_vulnerability(Branch, VulnerableVersion, Name, AppVersions) ->
          [{AppVersion :: binary(), Vulnerable :: binary()}] when
      Branch :: binary(),
      VulnerableVersion :: binary(),
      Name :: binary(),
      AppVersions :: binary().
get_otp_app_version_from_gh_vulnerability(BranchVersion, VulnerableVersion, Name, AppVersions) ->
    VulnerableVersion1 = parse_vulnerable_version_range_gh(BranchVersion, VulnerableVersion, Name),
    lists:uniq(
      [{AppVersion, VulnerableVersion1} ||

          %% split <<"4.15.3, 5.1.5, 5.2.9">> into multiple items
          AppVersion <- split_gh_version_binaries_into_list(AppVersions),

          %% fetch OTP versions attached to this item, e.g., 4.15.3 ==> ["26.2.5.6"]
          OTPVersion <- fetch_otp_major_version_from_table(<<Name/binary, "-", AppVersion/binary>>),

          %% if major version match, then accept them
          BranchVersion == hd(binary:split(list_to_binary(OTPVersion), ~".", [global]))]).


parse_vulnerable_version_range_gh(BranchVersion, <<">=", Version/binary>>, Name) ->
    case length(binary:split(Version, ~",", [global, trim_all])) of
        X when X > 1 ->
            %% this reported vulnerability does not follow a previous format
            %% and we cannot validate it.
            throw(not_valid_cve_report);
        X when X == 1 ->
            AppVersion = string:trim(Version),
            OTPVersion = fetch_otp_major_version_from_table(<<Name/binary, "-", AppVersion/binary>>),
            OTPMajorVersion = hd(binary:split(list_to_binary(OTPVersion), ~".", [global])),
            case BranchVersion == OTPMajorVersion of
                true ->
                    AppVersion;
                false ->
                    %% return first version from BranchVersion of the App.
                    Vuln = fetch_app_from_table(binary_to_list(<<"OTP-", BranchVersion/binary, ".0">>), Name),
                    [_, VulnVersion] = string:split(Vuln, ~"-"),
                    list_to_binary(VulnVersion)
            end
    end.

%% Input: <<"27.3.3, 26.2.5.11, 25.3.2.20">>
%% Output: [~"27.3.3", ~"26.2.5.11", ~"25.3.2.20">>]
-spec split_gh_version_binaries_into_list(binary()) -> [binary()].
split_gh_version_binaries_into_list(Bin) ->
    [string:trim(P) || P <- binary:split(Bin, ~",", [global, trim_all])].


extract_http_gh_body(RawHTTP) when is_list(RawHTTP) ->
    Body = lists:last(RawHTTP),
    json:decode(Body).

extract_http_gh_link(RawHTTP) when is_list(RawHTTP) ->
    lists:filtermap(fun(<<"Link: ", _/binary>>=Link) ->
                            Result = re:run(Link, "<([^>]+)>;\s*rel=\"next\"", [global, {capture, [1], list}]),
                            case Result of
                                {match, [[NextLink]]} ->
                                    [_, LinkPart] = string:split(NextLink, ~"?"),
                                    {true, generate_gh_link(LinkPart)};
                                _ ->
                                    false
                            end;
                       (_) ->
                            false
                    end, RawHTTP).

verify_advisory_against_openvex(OpenVEX, Advisory) ->
    AdvInfo = extract_advisory_info(Advisory),
    AdvVEX = extract_openvex_info(OpenVEX),

    %% checks that AdvVex is part of OpenVEX
    %% returns a list of missing OpenVEX statements into some branch.
    vex_set_inclusion(AdvInfo, AdvVEX).

%%
%% Extracts information from GH Advisories
%%
-spec extract_advisory_info(Advisories :: [map()]) -> [cve()].
extract_advisory_info(Advisories) when is_list(Advisories) ->
    lists:foldl(
      fun (Advisory, Acc) ->
              #{~"vulnerabilities" := Packages, ~"cve_id" := CVEId} = Advisory,
              lists:map(fun (#{~"package" := #{~"name" := AppName},
                               ~"patched_versions" := PatchedVersion,
                               ~"vulnerable_version_range" := AffectedVersion}) ->
                                create_cve(CVEId, AppName, [AffectedVersion], [PatchedVersion])
                        end, Packages) ++ Acc
      end, [], Advisories).

-spec create_cve(CVEId, AppName, PatchedVersions, AffectedVersions) -> cve() when
      CVEId :: binary(),
      AppName :: binary(),
      PatchedVersions :: [binary()],
      AffectedVersions :: [binary()].
create_cve(CVEId, AppName, PatchedVersions, AffectedVersions) ->
    #{'CVE' => CVEId,
      'appName' => AppName,
      'affectedVersions' => PatchedVersions,
      'fixedVersions' => AffectedVersions}.

%%
%% Extract information from OpenVEX statements
%%
-spec extract_openvex_info(OpenVEX :: map()) -> [cve()].
extract_openvex_info(#{~"statements" := Statements}) ->
    lists:foldl(fun (#{~"status" := Status}, Acc) when Status =:= ~"not_affected";
                                                       Status =:= ~"under_investigation" ->
                        Acc;
                    (#{~"status" := Status}=Vuln, Acc) when Status =:= ~"affected";
                                                            Status =:= ~"fixed" ->
                        CVEId = openvex_vuln_name(Vuln),
                        Products = openvex_vuln_products(Vuln),
                        case openvex_filter_product(Products) of
                            [] ->
                                Acc;
                            [{AppName, Versions}] ->
                                Found = lists:search(fun (#{'CVE' := CVE0,'appName' := AppName0}) ->
                                                          CVE0 == CVEId andalso AppName0 == AppName
                                                     end, Acc),
                                case Status of
                                    ~"affected" ->
                                        %% calculate < than using the format below
                                        %% [<<"26">>, <<"1">>] < [<<"26">>, <<"1">>, <<"0">>].
                                        VulnVersion = openvex_vuln_version(Versions, fun erlang:'=<'/2),
                                        case Found of
                                            false ->
                                                [create_cve(CVEId, AppName, [VulnVersion], []) | Acc];
                                            {value, #{'fixedVersions' := FixedVersions}=Item}  ->
                                                [create_cve(CVEId, AppName, [VulnVersion], FixedVersions) | (Acc -- [Item])]
                                        end;
                                    ~"fixed" ->
                                        VulnVersion = openvex_vuln_version(Versions, fun erlang:'>'/2),

                                        case Found of
                                            false ->
                                                [create_cve(CVEId, AppName, [], [VulnVersion]) | Acc];
                                            {value, #{'affectedVersions' := AffectedVersions}=Item}  ->
                                                [create_cve(CVEId, AppName, AffectedVersions, [VulnVersion]) | (Acc -- [Item])]
                                        end
                                end
                        end
                end, [], Statements).

openvex_vuln_name(#{~"vulnerability" := #{~"name" := Name}}) ->
    Name.

openvex_vuln_products(#{~"products" := Products}) ->
    Products.

openvex_vuln_version(Versions, Comparator) ->
    lists:foldl(fun (X, <<>>) -> X;
                    (X, Acc) ->
                        case Comparator(X, Acc) of
                            true ->
                                X;
                            false ->
                                Acc
                        end
                end, <<>>, Versions).


openvex_filter_product(Products) ->
    lists:foldl(fun (#{~"@id" := <<"pkg:otp/", Pkg/binary>>}, Acc) ->
                        [AppName, Version] = string:split(Pkg, ~"@"),
                        case Acc of
                            [] ->
                                [{AppName, [Version]}];
                            [{AppName, Versions}] ->
                                [{AppName, [Version | Versions]}]
                        end;
                    (_, Acc) -> Acc
                end, [], Products).

vex_set_inclusion(AdvVEX, OpenVEX) ->
    [VEX || VEX <- AdvVEX, not lists:member(VEX, OpenVEX)].

calculate_statements(VexStmts, VexTableFile, Branch, VexPath) ->
    VexTable = decode(VexTableFile),
    case maps:get(Branch, VexTable, error) of
        error ->
            fail("Could not find '~ts' in file '~ts'.~nDid you forget to add an entry with name '~ts' into 'openvex.table'?",
                 [Branch, VexTableFile, Branch]);
        CVEs ->
            calculate_statements_from_cves(VexStmts, CVEs, Branch, VexPath)
    end.

exists_cve_in_openvex(VexStmts, CVE, StatusCVE, Purl) ->
    lists:any(fun (#{~"vulnerability" := #{~"name" := VexCVE}}) when VexCVE =/= CVE ->
                      false;
                  (#{~"vulnerability" := #{~"name" := VexCVE}, ~"status" := Status}) ->
                    Ls = fetch_openvex_table_status(StatusCVE),
                    lists:member(Status, Ls) andalso CVE == VexCVE;
                  (#{~"products" := Products}) ->
                      VexIds = lists:map(fun(M0) -> maps:get(~"@id", M0) end, Products),
                      lists:member(Purl, VexIds)
              end, VexStmts).

fetch_openvex_table_status(#{~"affected" := _}=Status) when is_map(Status) ->
    [~"affected" | fetch_openvex_table_status(maps:without([~"affected"], Status))];
fetch_openvex_table_status(#{~"fixed" := _}=Status) when is_map(Status) ->
    [~"fixed" | fetch_openvex_table_status(maps:without([~"fixed"], Status))];
fetch_openvex_table_status(#{~"not_affected" := _}=Status) when is_map(Status) ->
    [~"not_affected" | fetch_openvex_table_status(maps:without([~"not_affected"], Status))];
fetch_openvex_table_status(Status) when Status == ~"under_investigation" ->
    [Status];
fetch_openvex_table_status(_) ->
    [].

fetch_openvex_status(M) when is_map(M) ->
    FixedStatus = maps:is_key(~"fixed", M),
    AffectedStatus = maps:is_key(~"affected", M),
    {FixedStatus, AffectedStatus};
fetch_openvex_status(_) ->
    {false, false}.

calculate_statements_from_cves(VexStmts, CVEs, Branch, VexPath) ->
    %% make the function idempotent, i.e., can be called consecutive times producing the same input
    lists:foldl(
      fun (#{~"status" := Status}=M, Acc) ->
              [{Purl, CVE}] = maps:to_list(maps:remove(~"status", M)),
              ExistingEntry = exists_cve_in_openvex(VexStmts, CVE, Status, Purl),
              case ExistingEntry of
                  true -> %% entry exists, ignore to make operation idempotent
                      Acc;
                  false ->
                      InitVex = vex_path(VexPath, Branch),
                      {FixedStatus, AffectedStatus} = fetch_openvex_status(Status),
                      case Purl of
                          <<?ErlangPURL, _/binary>> ->
                              case FixedStatus andalso AffectedStatus of
                                  true ->
                                      throw("Erlang/OTP release versions, (e.g.) OTP-26.1 do not support fixed and affected status");
                                  false ->
                                      [format_vexctl(InitVex, Purl, CVE, Status) | Acc]
                              end;
                          <<"pkg:otp/", _/binary>> -> % handle OTP Apps, pkg:otp/ssl@4.3.1
                              FixedRange =
                                  case FixedStatus orelse AffectedStatus of
                                      true ->
                                          maps:get(~"fixed", Status, []);
                                      _ ->
                                          %% not affected and we return all Erlang intermediate
                                          %% versions and all intermediate apps
                                          all
                                  end,
                              {OTPVersionsAffected, OTPVersionsFixed} = fetch_otp_purl_versions(Purl, FixedRange),
                              R = format_vexctl(InitVex, OTPVersionsAffected, OTPVersionsFixed, CVE, Status),
                              R ++ Acc;
                          _ -> % vendor
                              R = create_vendor_statements(FixedStatus andalso AffectedStatus, Status, InitVex, CVE, Purl),
                              R ++ Acc
                      end
              end
      end, [], CVEs).

create_vendor_statements(true, #{~"apps" := _},  _, _, _) ->
    %% this case is not accepted as input, e.g.
    %% the following is rejected
    %% {"pkg:github/madler/zlib@04f42ceca40f73e2978b50e93806c2a18c1281fc": "FIKA-2026-BROD",
    %%  "status": { "affected": "Mitigation message, update to the next release",
    %%              "fixed": ["pkg:github/madler/zlib@04f42thiscommitfixesthecve"],
    %%              "apps": ["pkg:otp/erts@14.2.5.10"]} }
    %% the current syntax from above has no way to understand when in erts this was fixed.
    %%
    %% If this case arises, write the CVE for zlib and then for OTP.
    fail("Case containing 'affected', 'fixed', and 'apps' (all three) not supported.", []);
create_vendor_statements(_, #{~"apps" := Apps}=Status, InitVex, CVE, Purl) ->
    {OTPVersionsAffected, OTPVersionsFixed} =
        lists:foldl(fun (App, {Af, Fx}) ->
                            {Affected, Fixed} = fetch_otp_purl_versions(App, all),
                            {merge_otp_version_binaries(Affected, Af),
                             merge_otp_version_binaries(Fixed, Fx)}
                    end, {<<>>, <<>>}, Apps),
    AppsR = format_vexctl(InitVex, OTPVersionsAffected, OTPVersionsFixed, CVE, Status),
    %% handle vendor dependencies. we lack sha-1 information to create
    %% a range of commits. if one wants to provide specific vendor information,
    %% e.g., false positive for openssl, one can do that manually using vexctl.
    %% if one wants to mention that erts-10.9.4 is not vulnerable to CVE-XXX
    %% in openssl, that's possible and goes via first case, pkg:otp/erts@10.9.4.
    FixedRange = maps:get(~"fixed", Status, <<>>),
    AppsR ++ format_vexctl(InitVex, Purl, FixedRange, CVE, Status);
create_vendor_statements(_, Status, InitVex, CVE, Purl) when is_map(Status) ->
    %% handle vendor dependencies. we lack sha-1 information to create
    %% a range of commits. if one wants to provide specific vendor information,
    %% e.g., false positive for openssl, one can do that manually using vexctl.
    %% if one wants to mention that erts-10.9.4 is not vulnerable to CVE-XXX
    %% in openssl, that's possible and goes via first case, pkg:otp/erts@10.9.4.
    FixedRange = maps:get(~"fixed", Status, <<>>),
    format_vexctl(InitVex, Purl, FixedRange, CVE, Status);
create_vendor_statements(_, Status,InitVex, CVE, Purl) when is_binary(Status) ->
    NotFixed = <<>>,
    format_vexctl(InitVex, Purl, NotFixed, CVE, Status).

format_vexctl(InitVex, Affected, Fixed, CVE, Status) ->
    Format = fun (X) -> case X of [] -> []; _ -> [X] end end,
    Format(format_vexctl(InitVex, Affected, CVE, Status)) ++
    Format(format_vexctl(InitVex, Fixed, CVE, ~"fixed")).


format_vexctl(_VexPath, <<>>, _CVE, _) ->
    [];
format_vexctl(VexPath, Versions, CVE, #{~"not_affected" := ~"vulnerable_code_not_present"}) ->
    io_lib:format("vexctl add --in-place ~ts --product='~ts' --vuln='~ts' --status='~ts' --justification='~ts'~n",
              [VexPath, Versions, CVE, ~"not_affected", ~"vulnerable_code_not_present"]);
format_vexctl(VexPath, Versions, CVE, #{~"affected" := Mitigation}) ->
    io_lib:format("vexctl add --in-place ~ts --product='~ts' --vuln='~ts' --status='~ts' --action-statement='~ts'~n",
          [VexPath, Versions, CVE, ~"affected", Mitigation]);
format_vexctl(VexPath, Versions, CVE, S) when S =:= ~"fixed";
                                              S =:= ~"under_investigation";
                                              S =:= ~"affected" ->
    io_lib:format("vexctl add --in-place ~ts --product='~ts' --vuln='~ts' --status='~ts'~n",
              [VexPath, Versions, CVE, S]).


-spec fetch_otp_purl_versions(OTP :: binary(), FixedVersions :: [binary()] ) ->
          {AffectedPurls :: binary(), FixedPurls :: binary()} | false.
fetch_otp_purl_versions(<<?ErlangPURL, _/binary>>, _FixedVersions) ->
    %% ignore
    false;
fetch_otp_purl_versions(<<"pkg:otp/", OTPApp/binary>>, all=_FixedVersions) ->
    %% Used to fetch all OTP releases and OTPApp versions
    %% starting from OTPApp Version

    AffectedVersions = fetch_version_from_table(OTPApp),
    ErlangOTPRelease = erlang:hd(AffectedVersions),
    {MajorVersion, _} = string:take(ErlangOTPRelease, ".", true, leading),

    All = fetch_otp_major_version_from_table(MajorVersion),
    RelevantVersions = take_otp_versions_from(All, AffectedVersions),

    {AffResult, FixResult} =
        lists:foldl(fun (V, {AffectedPurls, FixedPurls}) ->
                        All2 = fetch_app_from_table(V, OTPApp),
                        LastVersion = erlang:list_to_binary("pkg:otp/" ++ string:replace(All2, ~"-", ~"@")),
                        {AfP, FxP} = fetch_otp_purl_versions(LastVersion, []),
                        AfPResult = merge_otp_version_binaries(AfP, AffectedPurls),
                        FxPResult = merge_otp_version_binaries(FxP, FixedPurls),
                        {AfPResult, FxPResult}
                end, {<<>>, <<>>}, RelevantVersions),
    {AffResult, FixResult};
fetch_otp_purl_versions(<<"pkg:otp/", OTPApp/binary>>, FixedVersions) ->
    AffectedVersions = fetch_version_from_table(OTPApp),
    FixedRangeVersions = lists:flatmap(fun (<<"pkg:otp/", App/binary>>) ->
                                               fetch_version_from_table(App)
                                       end, FixedVersions),

    % Proceed to figure out OTP affected versions
    AffectedOTPVersionsInTree = calculate_otp_range_versions(AffectedVersions, FixedRangeVersions),
    OTPVersions = build_erlang_version_from_list(AffectedOTPVersionsInTree),
    OTPPurls = lists:map(fun erlang_purl/1, OTPVersions),
    AppVersions = lists:uniq(
                    lists:flatmap(fun (V) ->
                                          Apps = fetch_app_from_table(V, OTPApp),
                                          lists:map(fun (X) -> "pkg:otp/" ++ string:replace(X, ~"-", ~"@") end, Apps)
                                  end, OTPVersions)),

    AffectedPurls = erlang:list_to_binary(lists:join(",", OTPPurls ++ AppVersions)),

    % Proceed to create fixed versions
    FixedOTPVersions = lists:map(fun erlang_purl/1,
                                 build_erlang_version_from_list(otp_version_to_number(FixedRangeVersions))),
    FixedAppVersions = lists:map(fun erlang:binary_to_list/1, FixedVersions),
    FixedPurls = erlang:list_to_binary(lists:join(",", FixedOTPVersions ++ FixedAppVersions)),

    {AffectedPurls, FixedPurls};
fetch_otp_purl_versions(_, _) ->
    false.

erlang_purl(Release) when is_list(Release) ->
    ?ErlangPURL ++ "@OTP-" ++ Release.

take_otp_versions_from(Versions, AffectedVersions) ->
    F = fun (OTPRel) -> not lists:member(OTPRel, AffectedVersions) end,
    AffectedVersions ++ lists:takewhile(F, Versions).

merge_otp_version_binaries(A, B) ->
    case {A, B} of
        {<<>>, B} ->
            B;
        {_, <<>>} ->
            A;
        {_, _} ->
            remove_duplicate_versions(<<A/binary, ",", B/binary>>)
    end.

-spec remove_duplicate_versions(ListOfVulnerabilities :: binary()) -> binary().
remove_duplicate_versions(Version) ->
    binary:join(
      lists:uniq(
        binary:split(Version, ~",", [global])),
      <<",">>).


%% Versions = [ [26, 0], [26, 1, 2], ...  ] represents ["26.1", "26.1.2"]
build_erlang_version_from_list(Versions) ->
    lists:map(fun (X) ->
                      lists:join(".", lists:map(fun erlang:integer_to_list/1, X))
              end, Versions).

calculate_otp_range_versions(AffectedVersions, FixedRangeVersions) ->
    Vs = get_otp_version_tree(AffectedVersions),
    AffectedVersionsNumber = otp_version_to_number(AffectedVersions),
    FixedVersionsNumber = otp_version_to_number(FixedRangeVersions),
    Tree = build_tree(Vs),
    prune_trees(Tree, AffectedVersionsNumber, FixedVersionsNumber).

-spec build_tree(OTPTree :: list()) -> [{branch, Tree :: list()}].
build_tree(OTPTree) ->
    Sorted = lists:sort(fun less_than/2, OTPTree),
    Tree = build_tree(Sorted, 1, []),
    lists:map(fun ({branch, _}=Branch) -> Branch;
                  (Root) when is_list(Root) -> {branch, Root}
              end, Tree).

build_tree([], Pos, Acc) when Pos >= 4 ->
    {Acc, 0, []};
build_tree([], _Pos, Acc) ->
    [Acc];
build_tree([N| Ns], LastPos, Acc) when length(N) < 4, LastPos < 4 ->
    build_tree(Ns, length(N), [N | Acc]);
build_tree([N| Ns], LastPos, Acc) when length(N) >= 4, LastPos >= 4 ->
    build_tree(Ns, length(N), [N | Acc]);
build_tree([N| Ns], LastPos, Acc) when length(N) < 4, LastPos >= 4 ->
    {Acc, length(N), [N|Ns]};
build_tree([N | Ns], LastPos, Acc) when length(N) == 4, LastPos < 4  ->
    %% this is a new branch
    {Branch, N1, Continuation} = build_tree(Ns, length(N), [N | Acc]),
    [{branch, Branch} | build_tree(Continuation, N1, Acc)].


get_otp_version_tree(AffectedVersions) ->
      lists:uniq(
        lists:flatmap(fun (Version) ->
                              "OTP-"++Version1 = Version,
                              [Major|_] = convert_range(Version1),
                              OTPFlatTree = fetch_otp_major_version_from_table("OTP-"++Major),
                              lists:map(fun (X) ->
                                                lists:map(fun erlang:list_to_integer/1, convert_range(X))
                                        end, OTPFlatTree)
                      end, AffectedVersions)).

%% OTPVersion :: "OTP-26", e.g.
-spec otp_version_to_number(Ls) -> [Versions] when
      Ls :: [OTPVersion],
      OTPVersion :: string(),
      Versions :: string().
otp_version_to_number(Ls) ->
    lists:map(fun (X) ->
                      {_, Version} = string:take(string:trim(X, both), "OTP-"),
                      lists:map(fun erlang:list_to_integer/1, convert_range(Version))
              end, Ls).

prune_trees(Trees, AffectedVersions, FixedVersions) ->
    lists:sort(lists:uniq(
      lists:flatmap(fun({branch, Branch}) ->
                            Result = prune_tree(Branch, FixedVersions, lt),
                            prune_tree(Result, AffectedVersions, gt)
                    end, Trees) ++ AffectedVersions) -- FixedVersions).

%% assumption: list versions are sorted, as per otp_versions.
prune_tree(Ls, Affected, Comparator) ->
    Comp = case Comparator of
               lt -> true;
               gt -> false
           end,
    lists:uniq([L || A <:- Affected, lists:member(A, Ls), L <:- Ls, less_than(L, A) == Comp ]).

less_than([], []) ->
    true;
less_than([M | Ms], []) ->
    less_than([M | Ms], [0]);
less_than([], [N | Ns]) ->
    less_than([0], [ N | Ns]);
less_than([M | Ms], [N | Ns]) when M == N ->
    less_than(Ms, Ns);
less_than([M | _], [N | _]) when M =< N ->
    true;
less_than([M | _], [N | _]) when M > N ->
    false.

-spec fetch_version_from_table(OTPApp :: binary()) -> [string()].
fetch_version_from_table(OTPApp) ->
    App = erlang:list_to_binary(string:replace(OTPApp, ~"@", ~"-")),
    fetch_from_table(erlang:binary_to_list(App)).

-spec fetch_otp_major_version_from_table(Major :: string()) -> [string()].
fetch_otp_major_version_from_table(Major) when is_binary(Major)->
    fetch_otp_major_version_from_table(binary_to_list(Major));
fetch_otp_major_version_from_table(Major) when is_list(Major)->
    Ls = fetch_otp_from_version_table(Major),
    lists:map(fun ("OTP-"++Version) -> Version end, Ls).

fetch_from_table(Str) ->
    Vulns = os:cmd("grep '"++ Str ++ " ' otp_versions.table | cut -d' ' -f1"),
    lists:filter(fun (L) -> L=/= [] end, string:split(Vulns, ~"\n", all)).

fetch_otp_from_version_table(OTPVersion) ->
    Vulns = os:cmd("grep '"++ OTPVersion ++ "' otp_versions.table | cut -d' ' -f1"),
    lists:filter(fun (L) -> L=/= [] end, string:split(Vulns, ~"\n", all)).

%% OTPVersion = "OTP-26.3.1"
%% App = <<"ssl-XXXX">>
fetch_app_from_table(OTPVersion, App0) ->
    App = lists:takewhile(fun (Char) -> Char =/= $@ end, erlang:binary_to_list(App0)),
    Version = os:cmd("grep '" ++ OTPVersion ++ " : ' otp_versions.table"),
    Vulns = string:split(Version, ~" ", all),
    lists:filter(fun (L) ->
                         case string:prefix(L, App) of
                             nomatch ->
                                 false;
                             _ ->
                                 true
                         end
                 end, Vulns).

convert_range(Version) ->
    string:split(Version, ".", all).

%% Branch = "otp-28"
init_openvex_file(Branch) ->
    Ts = calendar:system_time_to_rfc3339(erlang:system_time(microsecond), [{unit, microsecond}]),
    [~"otp", Version] = string:split(Branch, ~"-"),
    #{
      ~"@context"   => ~"https://openvex.dev/ns/v0.2.0",
      ~"@id"        => openvex_iri(Version),
      ~"author"     => ~"vexctl",
      ~"timestamp"  => erlang:list_to_binary(Ts),
      ~"version"    => 1,
      ~"statements" => []
     }.

test_openvex(_) ->
    Tests = [
             test_openvex_branched_otp_tree,
             test_openvex_branched_otp_tree_idempotent
            ],
    lists:all(fun (Fun) ->
                      Result = with_otp_versions_table(Fun),
                      L = length(atom_to_list(Fun)),
                      io:format("- ~s~s~s~n", [Fun, lists:duplicate(80 - L, "."), Result]),
                      ok == Result
              end, Tests),
    ok.


test_openvex_branched_otp_tree() ->
    {VexPath,  Branch, VexStmts} = setup_openvex_test(),
    CVEs = fixup_openvex_branched_otp_tree(),
    Result = calculate_statements_from_cves(VexStmts, CVEs, Branch, VexPath),
    Expected = [~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/erlang/otp@OTP-23.0,pkg:github/erlang/otp@OTP-23.0.1,pkg:github/erlang/otp@OTP-23.0.2,pkg:github/erlang/otp@OTP-23.0.3,pkg:github/erlang/otp@OTP-23.0.4,pkg:otp/ssl@10.0,pkg:github/erlang/otp@OTP-23.1,pkg:github/erlang/otp@OTP-23.1.1,pkg:github/erlang/otp@OTP-23.1.2,pkg:github/erlang/otp@OTP-23.1.3,pkg:github/erlang/otp@OTP-23.1.4,pkg:github/erlang/otp@OTP-23.1.4.1,pkg:github/erlang/otp@OTP-23.1.5,pkg:otp/ssl@10.1,pkg:github/erlang/otp@OTP-23.2,pkg:github/erlang/otp@OTP-23.2.1,pkg:otp/ssl@10.2,pkg:github/erlang/otp@OTP-23.2.2,pkg:github/erlang/otp@OTP-23.2.3,pkg:otp/ssl@10.2.1,pkg:github/erlang/otp@OTP-23.2.4,pkg:otp/ssl@10.2.2,pkg:github/erlang/otp@OTP-23.2.5,pkg:github/erlang/otp@OTP-23.2.6,pkg:otp/ssl@10.2.3,pkg:github/erlang/otp@OTP-23.2.7,pkg:otp/ssl@10.2.4,pkg:github/erlang/otp@OTP-23.2.7.1,pkg:otp/ssl@10.2.4.1,pkg:github/erlang/otp@OTP-23.2.7.2,pkg:github/erlang/otp@OTP-23.2.7.3,pkg:otp/ssl@10.2.4.2,pkg:github/erlang/otp@OTP-23.2.7.4,pkg:otp/ssl@10.2.4.3,pkg:github/erlang/otp@OTP-23.2.7.5,pkg:otp/ssl@10.2.4.4,pkg:github/erlang/otp@OTP-23.3,pkg:github/erlang/otp@OTP-23.3.1,pkg:otp/ssl@10.3,pkg:github/erlang/otp@OTP-23.3.2,pkg:github/erlang/otp@OTP-23.3.3,pkg:github/erlang/otp@OTP-23.3.4,pkg:github/erlang/otp@OTP-23.3.4.1,pkg:otp/ssl@10.3.1,pkg:github/erlang/otp@OTP-23.3.4.2,pkg:github/erlang/otp@OTP-23.3.4.3,pkg:github/erlang/otp@OTP-23.3.4.4,pkg:otp/ssl@10.3.1.1,pkg:github/erlang/otp@OTP-23.3.4.5,pkg:github/erlang/otp@OTP-23.3.4.6,pkg:github/erlang/otp@OTP-23.3.4.7,pkg:github/erlang/otp@OTP-23.3.4.8,pkg:github/erlang/otp@OTP-23.3.4.9,pkg:github/erlang/otp@OTP-23.3.4.10,pkg:github/erlang/otp@OTP-23.3.4.11,pkg:github/erlang/otp@OTP-23.3.4.12,pkg:github/erlang/otp@OTP-23.3.4.13,pkg:github/erlang/otp@OTP-23.3.4.14,pkg:otp/ssl@10.3.1.2,pkg:github/erlang/otp@OTP-23.3.4.15,pkg:otp/ssl@10.3.1.3,pkg:github/erlang/otp@OTP-23.3.4.16,pkg:otp/ssl@10.3.1.4,pkg:github/erlang/otp@OTP-23.3.4.17,pkg:github/erlang/otp@OTP-23.3.4.18,pkg:github/erlang/otp@OTP-23.3.4.19,pkg:github/erlang/otp@OTP-23.3.4.20,pkg:otp/ssl@10.3.1.5' --vuln='F00' --status='under_investigation'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/erlang/otp@OTP-26.0,pkg:otp/erts@14.0,pkg:github/erlang/otp@OTP-26.0.1,pkg:otp/erts@14.0.1,pkg:github/erlang/otp@OTP-26.0.2,pkg:otp/erts@14.0.2,pkg:github/erlang/otp@OTP-26.1,pkg:github/erlang/otp@OTP-26.1.1,pkg:otp/erts@14.1,pkg:github/erlang/otp@OTP-26.1.2,pkg:otp/erts@14.1.1,pkg:github/erlang/otp@OTP-26.2,pkg:otp/erts@14.2,pkg:github/erlang/otp@OTP-26.2.1,pkg:otp/erts@14.2.1,pkg:github/erlang/otp@OTP-26.2.2,pkg:otp/erts@14.2.2,pkg:github/erlang/otp@OTP-26.2.3,pkg:otp/erts@14.2.3,pkg:github/erlang/otp@OTP-26.2.4,pkg:otp/erts@14.2.4,pkg:github/erlang/otp@OTP-26.2.5,pkg:otp/erts@14.2.5,pkg:github/erlang/otp@OTP-26.2.5.1,pkg:otp/erts@14.2.5.1,pkg:github/erlang/otp@OTP-26.2.5.2,pkg:otp/erts@14.2.5.2,pkg:github/erlang/otp@OTP-26.2.5.3,pkg:otp/erts@14.2.5.3,pkg:github/erlang/otp@OTP-26.2.5.4,pkg:github/erlang/otp@OTP-26.2.5.5,pkg:otp/erts@14.2.5.4,pkg:github/erlang/otp@OTP-26.2.5.6,pkg:otp/erts@14.2.5.5,pkg:github/erlang/otp@OTP-26.2.5.7,pkg:otp/erts@14.2.5.6,pkg:github/erlang/otp@OTP-26.2.5.8,pkg:otp/erts@14.2.5.7,pkg:github/erlang/otp@OTP-26.2.5.9,pkg:otp/erts@14.2.5.8,pkg:github/erlang/otp@OTP-26.2.5.10,pkg:github/erlang/otp@OTP-26.2.5.11,pkg:otp/erts@14.2.5.9,pkg:github/erlang/otp@OTP-26.2.5.12,pkg:github/erlang/otp@OTP-26.2.5.13,pkg:otp/erts@14.2.5.10,pkg:github/erlang/otp@OTP-26.2.5.14,pkg:github/erlang/otp@OTP-26.2.5.15,pkg:otp/erts@14.2.5.11' --vuln='CVE-2024-4444' --status='not_affected' --justification='vulnerable_code_not_present'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/openssl/openssl@0foobar' --vuln='CVE-2024-4444' --status='not_affected' --justification='vulnerable_code_not_present'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/erlang/otp@OTP-26.0,pkg:otp/erts@14.0,pkg:github/erlang/otp@OTP-26.0.1,pkg:otp/erts@14.0.1,pkg:github/erlang/otp@OTP-26.0.2,pkg:otp/erts@14.0.2,pkg:github/erlang/otp@OTP-26.1,pkg:github/erlang/otp@OTP-26.1.1,pkg:otp/erts@14.1,pkg:github/erlang/otp@OTP-26.1.2,pkg:otp/erts@14.1.1,pkg:github/erlang/otp@OTP-26.2,pkg:otp/erts@14.2,pkg:github/erlang/otp@OTP-26.2.1,pkg:otp/erts@14.2.1,pkg:github/erlang/otp@OTP-26.2.2,pkg:otp/erts@14.2.2,pkg:github/erlang/otp@OTP-26.2.3,pkg:otp/erts@14.2.3,pkg:github/erlang/otp@OTP-26.2.4,pkg:otp/erts@14.2.4,pkg:github/erlang/otp@OTP-26.2.5,pkg:otp/erts@14.2.5,pkg:github/erlang/otp@OTP-26.2.5.1,pkg:otp/erts@14.2.5.1,pkg:github/erlang/otp@OTP-26.2.5.2,pkg:otp/erts@14.2.5.2,pkg:github/erlang/otp@OTP-26.2.5.3,pkg:otp/erts@14.2.5.3,pkg:github/erlang/otp@OTP-26.2.5.4,pkg:github/erlang/otp@OTP-26.2.5.5,pkg:otp/erts@14.2.5.4,pkg:github/erlang/otp@OTP-26.2.5.6,pkg:otp/erts@14.2.5.5,pkg:github/erlang/otp@OTP-26.2.5.7,pkg:otp/erts@14.2.5.6,pkg:github/erlang/otp@OTP-26.2.5.8,pkg:otp/erts@14.2.5.7,pkg:github/erlang/otp@OTP-26.2.5.9,pkg:otp/erts@14.2.5.8,pkg:github/erlang/otp@OTP-26.2.5.10,pkg:github/erlang/otp@OTP-26.2.5.11,pkg:otp/erts@14.2.5.9,pkg:github/erlang/otp@OTP-26.2.5.12,pkg:github/erlang/otp@OTP-26.2.5.13,pkg:otp/erts@14.2.5.10,pkg:github/erlang/otp@OTP-26.2.5.14,pkg:github/erlang/otp@OTP-26.2.5.15,pkg:otp/erts@14.2.5.11' --vuln='CVE-2024-9143' --status='not_affected' --justification='vulnerable_code_not_present'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/openssl/openssl@0foobar' --vuln='CVE-2024-9143' --status='not_affected' --justification='vulnerable_code_not_present'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/madler/zlib@04f42ceca40f73e2978b50e93806c2a18c1281fc' --vuln='FIKA-2026-BROD' --status='affected' --action-statement='Mitigation message, update to the next release'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/PCRE2Project/pcre2@2dce7761b1831fd3f82a9c2bd5476259d945da4d' --vuln='CVE-2025-58050' --status='affected'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/erlang/otp@OTP-23.2.2,pkg:github/erlang/otp@OTP-23.2.3,pkg:github/erlang/otp@OTP-23.2.4,pkg:github/erlang/otp@OTP-23.2.5,pkg:github/erlang/otp@OTP-23.2.6,pkg:github/erlang/otp@OTP-23.2.7,pkg:github/erlang/otp@OTP-23.2.7.1,pkg:github/erlang/otp@OTP-23.3,pkg:github/erlang/otp@OTP-23.3.1,pkg:github/erlang/otp@OTP-23.3.2,pkg:github/erlang/otp@OTP-23.3.3,pkg:github/erlang/otp@OTP-23.3.4,pkg:github/erlang/otp@OTP-23.3.4.1,pkg:otp/ssl@10.2.1,pkg:otp/ssl@10.2.2,pkg:otp/ssl@10.2.3,pkg:otp/ssl@10.2.4,pkg:otp/ssl@10.2.4.1,pkg:otp/ssl@10.3,pkg:otp/ssl@10.3.1' --vuln='CVE-2025-26618' --status='affected' --action-statement='Update to the next version'\n",

                ~"vexctl add --in-place otp-23.openvex.json --product='pkg:github/erlang/otp@OTP-23.3.4.4,pkg:github/erlang/otp@OTP-23.3.4.3,pkg:github/erlang/otp@OTP-23.3.4.2,pkg:github/erlang/otp@OTP-23.2.7.3,pkg:github/erlang/otp@OTP-23.2.7.2,pkg:otp/ssl@10.3.1.1,pkg:otp/ssl@10.2.4.2' --vuln='CVE-2025-26618' --status='fixed'\n"
               ],
    TestFun = fun (R) -> lists:member(erlang:list_to_binary(R), Expected) end,
    true = lists:all(TestFun, Result),
    ok.

%% idempotent: script runs once. if run again, no new vex statements are introduced,
%% because there was no change.
test_openvex_branched_otp_tree_idempotent() ->
    {VexPath,  Branch, VexStmts} = setup_openvex_test(fixup_openvex_branched_otp_tree_stmts()),
    CVEs = fixup_openvex_branched_otp_tree(),
    Result = calculate_statements_from_cves(VexStmts, CVEs, Branch, VexPath),
    true = Result == [],
    ok.

setup_openvex_test() ->
    VexPath = ~"",
    Branch = ~"otp-23",
    VexStmts = [],
    {VexPath,  Branch, VexStmts}.
setup_openvex_test(Stmts) ->
    {VexPath, Branch, _} = setup_openvex_test(),
    {VexPath, Branch, Stmts}.


fixup_openvex_branched_otp_tree() ->
[ #{ ~"pkg:otp/ssl@10.2.1" => ~"CVE-2025-26618",
     ~"status" => #{ ~"affected" => ~"Update to the next version",
                     ~"fixed" => [~"pkg:otp/ssl@10.3.1.1", ~"pkg:otp/ssl@10.2.4.2"]} },

  #{ ~"pkg:github/madler/zlib@04f42ceca40f73e2978b50e93806c2a18c1281fc" => ~"FIKA-2026-BROD",
     ~"status" => #{ ~"affected" => ~"Mitigation message, update to the next release"}},

  #{ ~"pkg:github/openssl/openssl@0foobar" => ~"CVE-2024-9143",
      ~"status" => #{ ~"not_affected" => ~"vulnerable_code_not_present",
                      ~"apps" => [~"pkg:otp/erts@14.0"]}},

  #{ ~"pkg:github/openssl/openssl@0foobar" => ~"CVE-2024-4444",
     ~"status" => #{ ~"not_affected" => ~"vulnerable_code_not_present",
                     ~"apps" => [~"pkg:otp/erts@14.2.5.10"]}},

  #{~"pkg:github/PCRE2Project/pcre2@2dce7761b1831fd3f82a9c2bd5476259d945da4d" => ~"CVE-2025-58050",
    ~"status" => ~"affected"},

  #{ ~"pkg:otp/ssl@10.2.1" => ~"F00",
     ~"status" => ~"under_investigation" }

].


fixup_openvex_branched_otp_tree_stmts() ->
    [#{ ~"vulnerability"=>
            #{"name"=> ~"CVE-2025-26618"},
        ~"products"=>
            [
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.2"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.3"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.4"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.5"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.6"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.7"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.7.1"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.1"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.2"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.3"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.4"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.4.1"},
             #{~"@id"=> ~"pkg:otp/ssl@10.2.1"},
             #{~"@id"=> ~"pkg:otp/ssl@10.2.2"},
             #{~"@id"=> ~"pkg:otp/ssl@10.2.3"},
             #{~"@id"=> ~"pkg:otp/ssl@10.2.4"},
             #{~"@id"=> ~"pkg:otp/ssl@10.2.4.1"},
             #{~"@id"=> ~"pkg:otp/ssl@10.3"},
             #{~"@id"=> ~"pkg:otp/ssl@10.3.1"}
            ],
        ~"status"=> ~"affected",
        ~"action_statement"=> ~"Update to the next version"
      },
     #{ ~"vulnerability"=>
            #{~"name"=> ~"CVE-2025-26618"},
        ~"products"=>
            [
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.4.4"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.4.3"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.3.4.2"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.7.3"},
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.7.2"},
             #{~"@id"=> ~"pkg:otp/ssl@10.3.1.1"},
             #{~"@id"=> ~"pkg:otp/ssl@10.2.4.2"}
            ],
        ~"status"=> ~"fixed"
      },
     #{~"vulnerability"=>
           #{~"name"=> ~"FIKA-2026-BROD"},
       ~"products"=>
           [
            #{~"@id"=> ~"pkg:github/madler/zlib@04f42ceca40f73e2978b50e93806c2a18c1281fc"}
           ],
       ~"status"=> ~"affected",
       ~"action_statement"=> ~"Mitigation message, update to the next release"
      },
     #{ ~"vulnerability" =>
            #{ ~"name" => ~"CVE-2024-9143" },
        ~"timestamp" => ~"2025-08-19T13:18:05.434247759+02:00",
        ~"products" =>
            [
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.0"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.0.1"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.0.2"},
             #{~"@id" => ~"pkg:otp/erts@14.0"},
             #{~"@id" => ~"pkg:otp/erts@14.0.1"},
             #{~"@id" => ~"pkg:otp/erts@14.0.2"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.1"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.1.1"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.1.2"},
             #{~"@id" => ~"pkg:otp/erts@14.1"},
             #{~"@id" => ~"pkg:otp/erts@14.1.1"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.1"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.2"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.3"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.4"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.1"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.2"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.3"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.4"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.5"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.6"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.7"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.8"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.9"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.10"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.11"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.12"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.13"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.14"},
             #{~"@id" => ~"pkg:otp/erts@14.2"},
             #{~"@id" => ~"pkg:otp/erts@14.2.1"},
             #{~"@id" => ~"pkg:otp/erts@14.2.2"},
             #{~"@id" => ~"pkg:otp/erts@14.2.3"},
             #{~"@id" => ~"pkg:otp/erts@14.2.4"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.1"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.2"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.3"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.4"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.5"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.6"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.7"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.8"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.9"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.10"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.11"}
        ],
      ~"status" => ~"not_affected",
      ~"justification" => ~"vulnerable_code_not_present"
      },
     #{ ~"vulnerability" => #{ ~"name" => ~"CVE-2024-9143" },
        ~"timestamp" => ~"2025-08-19T13:18:23.396290497+02:00",
        ~"products" =>
            [
             #{ ~"@id" => ~"pkg:github/openssl/openssl@0foobar" }
            ],
        ~"status" => ~"not_affected",
        ~"justification" => ~"vulnerable_code_not_present" },
     #{ ~"vulnerability" =>
            #{ ~"name" => ~"CVE-2024-4444" },
        ~"timestamp" => ~"2025-08-19T13:18:05.434247759+02:00",
        ~"products" =>
            [#{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.14"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.11"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.12"},
             #{~"@id" => ~"pkg:github/erlang/otp@OTP-26.2.5.13"},
             #{~"@id" => ~"pkg:otp/erts@14.2.5.10"}],
      ~"status" => ~"not_affected",
      ~"justification" => ~"vulnerable_code_not_present"
      },
     #{ ~"vulnerability" => #{ ~"name" => ~"CVE-2024-4444" },
        ~"timestamp" => ~"2025-08-19T13:18:23.396290497+02:00",
        ~"products" =>
            [
             #{ ~"@id" => ~"pkg:github/openssl/openssl@0foobar" }
            ],
        ~"status" => ~"not_affected",
        ~"justification" => ~"vulnerable_code_not_present" },

     #{ ~"vulnerability"=> #{"name"=> ~"F00"},
        ~"products"=>
            [
             #{~"@id"=> ~"pkg:github/erlang/otp@OTP-23.2.2"},
             #{~"@id"=> ~"pkg:otp/ssl@10.2.1"}
            ],
        ~"status"=> ~"under_investigation"
      },
     #{ ~"vulnerability"=> #{"name"=> ~"CVE-2025-58050"},
        ~"products"=>
            [
             #{~"@id"=> ~"pkg:github/PCRE2Project/pcre2@2dce7761b1831fd3f82a9c2bd5476259d945da4d"}
            ],
        ~"status"=> ~"affected"
      }
    ].

%% This table is used as fixed up data for the openvex verification.
with_otp_versions_table(F) ->
    OTPTable =
     """
     OTP-28.0.4 : inets-9.4.1 # asn1-5.4.1 common_test-1.28 compiler-9.0.1 crypto-5.6 debugger-6.0.2 dialyzer-5.4 diameter-2.5.1 edoc-1.4 eldap-1.2.16 erl_interface-5.6 erts-16.0.3 et-1.7.2 eunit-2.10 ftp-1.2.4 jinterface-1.15 kernel-10.3.2 megaco-4.8 mnesia-4.24 observer-2.18 odbc-2.16 os_mon-2.11 parsetools-2.7 public_key-1.18.2 reltool-1.0.2 runtime_tools-2.2 sasl-4.3 snmp-5.19 ssh-5.3.3 ssl-11.3.2 stdlib-7.0.3 syntax_tools-4.0 tftp-1.2.3 tools-4.1.2 wx-2.5.1 xmerl-2.1.5 :
     OTP-28.0.3 : diameter-2.5.1 erts-16.0.3 ssh-5.3.3 stdlib-7.0.3 # asn1-5.4.1 common_test-1.28 compiler-9.0.1 crypto-5.6 debugger-6.0.2 dialyzer-5.4 edoc-1.4 eldap-1.2.16 erl_interface-5.6 et-1.7.2 eunit-2.10 ftp-1.2.4 inets-9.4 jinterface-1.15 kernel-10.3.2 megaco-4.8 mnesia-4.24 observer-2.18 odbc-2.16 os_mon-2.11 parsetools-2.7 public_key-1.18.2 reltool-1.0.2 runtime_tools-2.2 sasl-4.3 snmp-5.19 ssl-11.3.2 syntax_tools-4.0 tftp-1.2.3 tools-4.1.2 wx-2.5.1 xmerl-2.1.5 :
     OTP-28.0.2 : compiler-9.0.1 debugger-6.0.2 erts-16.0.2 kernel-10.3.2 public_key-1.18.2 ssh-5.3.2 ssl-11.3.2 stdlib-7.0.2 wx-2.5.1 # asn1-5.4.1 common_test-1.28 crypto-5.6 dialyzer-5.4 diameter-2.5 edoc-1.4 eldap-1.2.16 erl_interface-5.6 et-1.7.2 eunit-2.10 ftp-1.2.4 inets-9.4 jinterface-1.15 megaco-4.8 mnesia-4.24 observer-2.18 odbc-2.16 os_mon-2.11 parsetools-2.7 reltool-1.0.2 runtime_tools-2.2 sasl-4.3 snmp-5.19 syntax_tools-4.0 tftp-1.2.3 tools-4.1.2 xmerl-2.1.5 :
     OTP-28.0.1 : asn1-5.4.1 debugger-6.0.1 eldap-1.2.16 erts-16.0.1 kernel-10.3.1 public_key-1.18.1 ssh-5.3.1 ssl-11.3.1 stdlib-7.0.1 xmerl-2.1.5 # common_test-1.28 compiler-9.0 crypto-5.6 dialyzer-5.4 diameter-2.5 edoc-1.4 erl_interface-5.6 et-1.7.2 eunit-2.10 ftp-1.2.4 inets-9.4 jinterface-1.15 megaco-4.8 mnesia-4.24 observer-2.18 odbc-2.16 os_mon-2.11 parsetools-2.7 reltool-1.0.2 runtime_tools-2.2 sasl-4.3 snmp-5.19 syntax_tools-4.0 tftp-1.2.3 tools-4.1.2 wx-2.5 :
     OTP-28.0 : asn1-5.4 common_test-1.28 compiler-9.0 crypto-5.6 debugger-6.0 dialyzer-5.4 diameter-2.5 edoc-1.4 eldap-1.2.15 erl_interface-5.6 erts-16.0 et-1.7.2 eunit-2.10 ftp-1.2.4 inets-9.4 jinterface-1.15 kernel-10.3 megaco-4.8 mnesia-4.24 observer-2.18 odbc-2.16 os_mon-2.11 parsetools-2.7 public_key-1.18 reltool-1.0.2 runtime_tools-2.2 sasl-4.3 snmp-5.19 ssh-5.3 ssl-11.3 stdlib-7.0 syntax_tools-4.0 tftp-1.2.3 tools-4.1.2 wx-2.5 xmerl-2.1.4 # :
     OTP-27.3.4.3 : compiler-8.6.1.2 debugger-5.5.0.1 erts-15.2.7.2 inets-9.3.2.1 ssh-5.2.11.3 syntax_tools-3.2.2.1 # asn1-5.3.4.2 common_test-1.27.7 crypto-5.5.3 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14.1 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 jinterface-1.14.1 kernel-10.2.7.2 megaco-4.7.2 mnesia-4.23.5 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18.2 ssl-11.2.12.2 stdlib-6.2.2.2 tftp-1.2.2 tools-4.1.1 wx-2.4.3 xmerl-2.1.3.1 :
     OTP-27.3.4.2 : asn1-5.3.4.2 compiler-8.6.1.1 erts-15.2.7.1 kernel-10.2.7.2 public_key-1.17.1.1 ssh-5.2.11.2 ssl-11.2.12.2 stdlib-6.2.2.2 # common_test-1.27.7 crypto-5.5.3 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14.1 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 megaco-4.7.2 mnesia-4.23.5 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18.2 syntax_tools-3.2.2 tftp-1.2.2 tools-4.1.1 wx-2.4.3 xmerl-2.1.3.1 :
     OTP-27.3.4.1 : asn1-5.3.4.1 eldap-1.2.14.1 kernel-10.2.7.1 ssh-5.2.11.1 ssl-11.2.12.1 stdlib-6.2.2.1 xmerl-2.1.3.1 # common_test-1.27.7 compiler-8.6.1 crypto-5.5.3 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 erl_interface-5.5.2 erts-15.2.7 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 megaco-4.7.2 mnesia-4.23.5 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18.2 syntax_tools-3.2.2 tftp-1.2.2 tools-4.1.1 wx-2.4.3 :
     OTP-27.3.4 : erts-15.2.7 kernel-10.2.7 ssh-5.2.11 xmerl-2.1.3 # asn1-5.3.4 common_test-1.27.7 compiler-8.6.1 crypto-5.5.3 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 megaco-4.7.2 mnesia-4.23.5 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18.2 ssl-11.2.12 stdlib-6.2.2 syntax_tools-3.2.2 tftp-1.2.2 tools-4.1.1 wx-2.4.3 :
     OTP-27.3.3 : erts-15.2.6 kernel-10.2.6 megaco-4.7.2 ssh-5.2.10 ssl-11.2.12 # asn1-5.3.4 common_test-1.27.7 compiler-8.6.1 crypto-5.5.3 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 mnesia-4.23.5 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18.2 stdlib-6.2.2 syntax_tools-3.2.2 tftp-1.2.2 tools-4.1.1 wx-2.4.3 xmerl-2.1.2 :
     OTP-27.3.2 : asn1-5.3.4 compiler-8.6.1 erts-15.2.5 kernel-10.2.5 megaco-4.7.1 snmp-5.18.2 ssl-11.2.11 xmerl-2.1.2 # common_test-1.27.7 crypto-5.5.3 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 mnesia-4.23.5 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 ssh-5.2.9 stdlib-6.2.2 syntax_tools-3.2.2 tftp-1.2.2 tools-4.1.1 wx-2.4.3 :
     OTP-27.3.1 : asn1-5.3.3 erts-15.2.4 kernel-10.2.4 mnesia-4.23.5 ssh-5.2.9 ssl-11.2.10 stdlib-6.2.2 # common_test-1.27.7 compiler-8.6 crypto-5.5.3 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 megaco-4.7 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18.1 syntax_tools-3.2.2 tftp-1.2.2 tools-4.1.1 wx-2.4.3 xmerl-2.1.1 :
     OTP-27.3 : asn1-5.3.2 common_test-1.27.7 compiler-8.6 crypto-5.5.3 erts-15.2.3 kernel-10.2.3 mnesia-4.23.4 ssh-5.2.8 ssl-11.2.9 stdlib-6.2.1 syntax_tools-3.2.2 xmerl-2.1.1 # debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 megaco-4.7 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18.1 tftp-1.2.2 tools-4.1.1 wx-2.4.3 :
     OTP-27.2.4 : snmp-5.18.1 ssh-5.2.7 # asn1-5.3.1 common_test-1.27.6 compiler-8.5.5 crypto-5.5.2 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 erts-15.2.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.2 jinterface-1.14.1 kernel-10.2.2 megaco-4.7 mnesia-4.23.3 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 ssl-11.2.8 stdlib-6.2 syntax_tools-3.2.1 tftp-1.2.2 tools-4.1.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.2.3 : inets-9.3.2 ssl-11.2.8 # asn1-5.3.1 common_test-1.27.6 compiler-8.5.5 crypto-5.5.2 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 erts-15.2.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 jinterface-1.14.1 kernel-10.2.2 megaco-4.7 mnesia-4.23.3 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17.1 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18 ssh-5.2.6 stdlib-6.2 syntax_tools-3.2.1 tftp-1.2.2 tools-4.1.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.2.2 : compiler-8.5.5 erts-15.2.2 kernel-10.2.2 public_key-1.17.1 ssl-11.2.7 # asn1-5.3.1 common_test-1.27.6 crypto-5.5.2 debugger-5.5 dialyzer-5.3.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.1 jinterface-1.14.1 megaco-4.7 mnesia-4.23.3 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18 ssh-5.2.6 stdlib-6.2 syntax_tools-3.2.1 tftp-1.2.2 tools-4.1.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.2.1 : common_test-1.27.6 dialyzer-5.3.1 erts-15.2.1 kernel-10.2.1 ssh-5.2.6 tftp-1.2.2 # asn1-5.3.1 compiler-8.5.4 crypto-5.5.2 debugger-5.5 diameter-2.4.1 edoc-1.3.2 eldap-1.2.14 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3.1 jinterface-1.14.1 megaco-4.7 mnesia-4.23.3 observer-2.17 odbc-2.15 os_mon-2.10.1 parsetools-2.6 public_key-1.17 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.18 ssl-11.2.6 stdlib-6.2 syntax_tools-3.2.1 tools-4.1.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.2 : common_test-1.27.5 compiler-8.5.4 crypto-5.5.2 debugger-5.5 dialyzer-5.3 eldap-1.2.14 erts-15.2 inets-9.3.1 kernel-10.2 megaco-4.7 mnesia-4.23.3 observer-2.17 os_mon-2.10.1 public_key-1.17 snmp-5.18 ssh-5.2.5 ssl-11.2.6 stdlib-6.2 tools-4.1.1 # asn1-5.3.1 diameter-2.4.1 edoc-1.3.2 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 jinterface-1.14.1 odbc-2.15 parsetools-2.6 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 syntax_tools-3.2.1 tftp-1.2.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.1.3 : common_test-1.27.4 compiler-8.5.3 erts-15.1.3 kernel-10.1.2 public_key-1.16.4 ssh-5.2.4 ssl-11.2.5 # asn1-5.3.1 crypto-5.5.1 debugger-5.4 dialyzer-5.2.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.13 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3 jinterface-1.14.1 megaco-4.6 mnesia-4.23.2 observer-2.16 odbc-2.15 os_mon-2.10 parsetools-2.6 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.17 stdlib-6.1.2 syntax_tools-3.2.1 tftp-1.2.1 tools-4.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.1.2 : common_test-1.27.3 erts-15.1.2 kernel-10.1.1 ssh-5.2.3 ssl-11.2.4 stdlib-6.1.2 # asn1-5.3.1 compiler-8.5.2 crypto-5.5.1 debugger-5.4 dialyzer-5.2.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.13 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3 jinterface-1.14.1 megaco-4.6 mnesia-4.23.2 observer-2.16 odbc-2.15 os_mon-2.10 parsetools-2.6 public_key-1.16.3 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.17 syntax_tools-3.2.1 tftp-1.2.1 tools-4.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.1.1 : common_test-1.27.2 erts-15.1.1 public_key-1.16.3 ssl-11.2.3 stdlib-6.1.1 # asn1-5.3.1 compiler-8.5.2 crypto-5.5.1 debugger-5.4 dialyzer-5.2.1 diameter-2.4.1 edoc-1.3.2 eldap-1.2.13 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.3 inets-9.3 jinterface-1.14.1 kernel-10.1 megaco-4.6 mnesia-4.23.2 observer-2.16 odbc-2.15 os_mon-2.10 parsetools-2.6 reltool-1.0.1 runtime_tools-2.1.1 sasl-4.2.2 snmp-5.17 ssh-5.2.2 syntax_tools-3.2.1 tftp-1.2.1 tools-4.1 wx-2.4.3 xmerl-2.1 :
     OTP-27.1 : asn1-5.3.1 common_test-1.27.1 compiler-8.5.2 crypto-5.5.1 dialyzer-5.2.1 diameter-2.4.1 edoc-1.3.2 erts-15.1 ftp-1.2.3 inets-9.3 kernel-10.1 odbc-2.15 public_key-1.16.2 runtime_tools-2.1.1 snmp-5.17 ssh-5.2.2 ssl-11.2.2 stdlib-6.1 syntax_tools-3.2.1 tftp-1.2.1 tools-4.1 wx-2.4.3 xmerl-2.1 # debugger-5.4 eldap-1.2.13 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 jinterface-1.14.1 megaco-4.6 mnesia-4.23.2 observer-2.16 os_mon-2.10 parsetools-2.6 reltool-1.0.1 sasl-4.2.2 :
     OTP-27.0.1 : compiler-8.5.1 edoc-1.3.1 erts-15.0.1 kernel-10.0.1 public_key-1.16.1 ssh-5.2.1 ssl-11.2.1 stdlib-6.0.1 # asn1-5.3 common_test-1.27 crypto-5.5 debugger-5.4 dialyzer-5.2 diameter-2.4 eldap-1.2.13 erl_interface-5.5.2 et-1.7.1 eunit-2.9.1 ftp-1.2.2 inets-9.2 jinterface-1.14.1 megaco-4.6 mnesia-4.23.2 observer-2.16 odbc-2.14.3 os_mon-2.10 parsetools-2.6 reltool-1.0.1 runtime_tools-2.1 sasl-4.2.2 snmp-5.16 syntax_tools-3.2 tftp-1.2 tools-4.0 wx-2.4.2 xmerl-2.0 :
     OTP-27.0 : asn1-5.3 common_test-1.27 compiler-8.5 crypto-5.5 debugger-5.4 dialyzer-5.2 diameter-2.4 edoc-1.3 eldap-1.2.13 erl_interface-5.5.2 erts-15.0 et-1.7.1 eunit-2.9.1 ftp-1.2.2 inets-9.2 jinterface-1.14.1 kernel-10.0 megaco-4.6 mnesia-4.23.2 observer-2.16 odbc-2.14.3 os_mon-2.10 parsetools-2.6 public_key-1.16 reltool-1.0.1 runtime_tools-2.1 sasl-4.2.2 snmp-5.16 ssh-5.2 ssl-11.2 stdlib-6.0 syntax_tools-3.2 tftp-1.2 tools-4.0 wx-2.4.2 xmerl-2.0 # :
     OTP-26.2.5.15 : inets-9.1.0.3 ssh-5.1.4.12 # asn1-5.2.2.1 common_test-1.26.2.4 compiler-8.4.3.3 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 erts-14.2.5.11 et-1.7 eunit-2.9 ftp-1.2.1.1 jinterface-1.14 kernel-9.2.4.10 megaco-4.5 mnesia-4.23.1.2 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.6 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssl-11.1.4.9 stdlib-5.2.3.5 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.3 :
     OTP-26.2.5.14 : erts-14.2.5.11 kernel-9.2.4.10 public_key-1.15.1.6 ssh-5.1.4.11 ssl-11.1.4.9 stdlib-5.2.3.5 # asn1-5.2.2.1 common_test-1.26.2.4 compiler-8.4.3.3 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 megaco-4.5 mnesia-4.23.1.2 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.3 :
     OTP-26.2.5.13 : asn1-5.2.2.1 kernel-9.2.4.9 ssh-5.1.4.10 stdlib-5.2.3.4 # common_test-1.26.2.4 compiler-8.4.3.3 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 erts-14.2.5.10 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 megaco-4.5 mnesia-4.23.1.2 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssl-11.1.4.8 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.3 :
     OTP-26.2.5.12 : compiler-8.4.3.3 erts-14.2.5.10 kernel-9.2.4.8 ssh-5.1.4.9 xmerl-1.3.34.3 # asn1-5.2.2 common_test-1.26.2.4 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 megaco-4.5 mnesia-4.23.1.2 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssl-11.1.4.8 stdlib-5.2.3.3 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 :
     OTP-26.2.5.11 : ssh-5.1.4.8 xmerl-1.3.34.2 # asn1-5.2.2 common_test-1.26.2.4 compiler-8.4.3.2 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 erts-14.2.5.9 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 kernel-9.2.4.7 megaco-4.5 mnesia-4.23.1.2 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssl-11.1.4.8 stdlib-5.2.3.3 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 :
     OTP-26.2.5.10 : erts-14.2.5.9 kernel-9.2.4.7 mnesia-4.23.1.2 ssh-5.1.4.7 ssl-11.1.4.8 # asn1-5.2.2 common_test-1.26.2.4 compiler-8.4.3.2 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 megaco-4.5 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 stdlib-5.2.3.3 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.1 :
     OTP-26.2.5.9 : erts-14.2.5.8 ssh-5.1.4.6 # asn1-5.2.2 common_test-1.26.2.4 compiler-8.4.3.2 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 kernel-9.2.4.6 megaco-4.5 mnesia-4.23.1.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssl-11.1.4.7 stdlib-5.2.3.3 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.1 :
     OTP-26.2.5.8 : erts-14.2.5.7 kernel-9.2.4.6 public_key-1.15.1.5 # asn1-5.2.2 common_test-1.26.2.4 compiler-8.4.3.2 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3.1 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 megaco-4.5 mnesia-4.23.1.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssh-5.1.4.5 ssl-11.1.4.7 stdlib-5.2.3.3 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.1 :
     OTP-26.2.5.7 : common_test-1.26.2.4 dialyzer-5.1.3.1 erts-14.2.5.6 kernel-9.2.4.5 ssh-5.1.4.5 ssl-11.1.4.7 # asn1-5.2.2 compiler-8.4.3.2 crypto-5.4.2.3 debugger-5.3.4 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.2 jinterface-1.14 megaco-4.5 mnesia-4.23.1.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.4 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 stdlib-5.2.3.3 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.1 :
     OTP-26.2.5.6 : common_test-1.26.2.3 erts-14.2.5.5 inets-9.1.0.2 kernel-9.2.4.4 mnesia-4.23.1.1 public_key-1.15.1.4 ssl-11.1.4.6 stdlib-5.2.3.3 # asn1-5.2.2 compiler-8.4.3.2 crypto-5.4.2.3 debugger-5.3.4 dialyzer-5.1.3 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 jinterface-1.14 megaco-4.5 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssh-5.1.4.4 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.1 :
     OTP-26.2.5.5 : common_test-1.26.2.2 crypto-5.4.2.3 ssh-5.1.4.4 ssl-11.1.4.5 # asn1-5.2.2 compiler-8.4.3.2 debugger-5.3.4 dialyzer-5.1.3 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 erts-14.2.5.4 et-1.7 eunit-2.9 ftp-1.2.1.1 inets-9.1.0.1 jinterface-1.14 kernel-9.2.4.3 megaco-4.5 mnesia-4.23.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.3 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 stdlib-5.2.3.2 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34.1 :
     OTP-26.2.5.4 : common_test-1.26.2.1 compiler-8.4.3.2 crypto-5.4.2.2 erts-14.2.5.4 inets-9.1.0.1 kernel-9.2.4.3 public_key-1.15.1.3 ssh-5.1.4.3 ssl-11.1.4.4 stdlib-5.2.3.2 xmerl-1.3.34.1 # asn1-5.2.2 debugger-5.3.4 dialyzer-5.1.3 diameter-2.3.2.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1.1 jinterface-1.14 megaco-4.5 mnesia-4.23.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 :
     OTP-26.2.5.3 : compiler-8.4.3.1 diameter-2.3.2.2 erts-14.2.5.3 ftp-1.2.1.1 kernel-9.2.4.2 public_key-1.15.1.2 ssh-5.1.4.2 ssl-11.1.4.3 # asn1-5.2.2 common_test-1.26.2 crypto-5.4.2.1 debugger-5.3.4 dialyzer-5.1.3 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 inets-9.1 jinterface-1.14 megaco-4.5 mnesia-4.23.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 stdlib-5.2.3.1 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34 :
     OTP-26.2.5.2 : crypto-5.4.2.1 erts-14.2.5.2 ssl-11.1.4.2 stdlib-5.2.3.1 # asn1-5.2.2 common_test-1.26.2 compiler-8.4.3 debugger-5.3.4 dialyzer-5.1.3 diameter-2.3.2.1 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1 inets-9.1 jinterface-1.14 kernel-9.2.4.1 megaco-4.5 mnesia-4.23.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1.1 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssh-5.1.4.1 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34 :
     OTP-26.2.5.1 : diameter-2.3.2.1 erts-14.2.5.1 kernel-9.2.4.1 public_key-1.15.1.1 ssh-5.1.4.1 ssl-11.1.4.1 # asn1-5.2.2 common_test-1.26.2 compiler-8.4.3 crypto-5.4.2 debugger-5.3.4 dialyzer-5.1.3 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1 inets-9.1 jinterface-1.14 megaco-4.5 mnesia-4.23.1 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 stdlib-5.2.3 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34 :
     OTP-26.2.5 : dialyzer-5.1.3 erts-14.2.5 kernel-9.2.4 mnesia-4.23.1 ssl-11.1.4 stdlib-5.2.3 # asn1-5.2.2 common_test-1.26.2 compiler-8.4.3 crypto-5.4.2 debugger-5.3.4 diameter-2.3.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1 inets-9.1 jinterface-1.14 megaco-4.5 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssh-5.1.4 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34 :
     OTP-26.2.4 : asn1-5.2.2 common_test-1.26.2 compiler-8.4.3 crypto-5.4.2 debugger-5.3.4 diameter-2.3.2 erts-14.2.4 kernel-9.2.3 ssh-5.1.4 ssl-11.1.3 stdlib-5.2.2 # dialyzer-5.1.2 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1 inets-9.1 jinterface-1.14 megaco-4.5 mnesia-4.23 observer-2.15.1 odbc-2.14.2 os_mon-2.9.1 parsetools-2.5 public_key-1.15.1 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4.1 xmerl-1.3.34 :
     OTP-26.2.3 : compiler-8.4.2 crypto-5.4.1 erts-14.2.3 kernel-9.2.2 odbc-2.14.2 public_key-1.15.1 ssh-5.1.3 ssl-11.1.2 stdlib-5.2.1 wx-2.4.1 # asn1-5.2.1 common_test-1.26.1 debugger-5.3.3 dialyzer-5.1.2 diameter-2.3.1 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5.1 et-1.7 eunit-2.9 ftp-1.2.1 inets-9.1 jinterface-1.14 megaco-4.5 mnesia-4.23 observer-2.15.1 os_mon-2.9.1 parsetools-2.5 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 syntax_tools-3.1 tftp-1.1.1 tools-3.6 xmerl-1.3.34 :
     OTP-26.2.2 : common_test-1.26.1 erl_interface-5.5.1 erts-14.2.2 kernel-9.2.1 ssh-5.1.2 ssl-11.1.1 # asn1-5.2.1 compiler-8.4.1 crypto-5.4 debugger-5.3.3 dialyzer-5.1.2 diameter-2.3.1 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 et-1.7 eunit-2.9 ftp-1.2.1 inets-9.1 jinterface-1.14 megaco-4.5 mnesia-4.23 observer-2.15.1 odbc-2.14.1 os_mon-2.9.1 parsetools-2.5 public_key-1.15 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 stdlib-5.2 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4 xmerl-1.3.34 :
     OTP-26.2.1 : erts-14.2.1 ssh-5.1.1 # asn1-5.2.1 common_test-1.26 compiler-8.4.1 crypto-5.4 debugger-5.3.3 dialyzer-5.1.2 diameter-2.3.1 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5 et-1.7 eunit-2.9 ftp-1.2.1 inets-9.1 jinterface-1.14 kernel-9.2 megaco-4.5 mnesia-4.23 observer-2.15.1 odbc-2.14.1 os_mon-2.9.1 parsetools-2.5 public_key-1.15 reltool-1.0 runtime_tools-2.0.1 sasl-4.2.1 snmp-5.15 ssl-11.1 stdlib-5.2 syntax_tools-3.1 tftp-1.1.1 tools-3.6 wx-2.4 xmerl-1.3.34 :
     OTP-26.2 : asn1-5.2.1 common_test-1.26 crypto-5.4 debugger-5.3.3 dialyzer-5.1.2 diameter-2.3.1 edoc-1.2.1 eldap-1.2.12 erl_docgen-1.5.2 erl_interface-5.5 erts-14.2 eunit-2.9 ftp-1.2.1 inets-9.1 kernel-9.2 mnesia-4.23 os_mon-2.9.1 public_key-1.15 runtime_tools-2.0.1 ssh-5.1 ssl-11.1 stdlib-5.2 tftp-1.1.1 wx-2.4 xmerl-1.3.34 # compiler-8.4.1 et-1.7 jinterface-1.14 megaco-4.5 observer-2.15.1 odbc-2.14.1 parsetools-2.5 reltool-1.0 sasl-4.2.1 snmp-5.15 syntax_tools-3.1 tools-3.6 :
     OTP-26.1.2 : erts-14.1.1 xmerl-1.3.33 # asn1-5.2 common_test-1.25.1 compiler-8.4.1 crypto-5.3 debugger-5.3.2 dialyzer-5.1.1 diameter-2.3 edoc-1.2 eldap-1.2.11 erl_docgen-1.5.1 erl_interface-5.4 et-1.7 eunit-2.8.2 ftp-1.2 inets-9.0.2 jinterface-1.14 kernel-9.1 megaco-4.5 mnesia-4.22.1 observer-2.15.1 odbc-2.14.1 os_mon-2.9 parsetools-2.5 public_key-1.14.1 reltool-1.0 runtime_tools-2.0 sasl-4.2.1 snmp-5.15 ssh-5.0.1 ssl-11.0.3 stdlib-5.1.1 syntax_tools-3.1 tftp-1.1 tools-3.6 wx-2.3.1 :
     OTP-26.1.1 : compiler-8.4.1 stdlib-5.1.1 wx-2.3.1 # asn1-5.2 common_test-1.25.1 crypto-5.3 debugger-5.3.2 dialyzer-5.1.1 diameter-2.3 edoc-1.2 eldap-1.2.11 erl_docgen-1.5.1 erl_interface-5.4 erts-14.1 et-1.7 eunit-2.8.2 ftp-1.2 inets-9.0.2 jinterface-1.14 kernel-9.1 megaco-4.5 mnesia-4.22.1 observer-2.15.1 odbc-2.14.1 os_mon-2.9 parsetools-2.5 public_key-1.14.1 reltool-1.0 runtime_tools-2.0 sasl-4.2.1 snmp-5.15 ssh-5.0.1 ssl-11.0.3 syntax_tools-3.1 tftp-1.1 tools-3.6 xmerl-1.3.32 :
     OTP-26.1 : asn1-5.2 common_test-1.25.1 compiler-8.4 crypto-5.3 debugger-5.3.2 dialyzer-5.1.1 erl_docgen-1.5.1 erts-14.1 inets-9.0.2 kernel-9.1 megaco-4.5 mnesia-4.22.1 observer-2.15.1 public_key-1.14.1 snmp-5.15 ssl-11.0.3 stdlib-5.1 # diameter-2.3 edoc-1.2 eldap-1.2.11 erl_interface-5.4 et-1.7 eunit-2.8.2 ftp-1.2 jinterface-1.14 odbc-2.14.1 os_mon-2.9 parsetools-2.5 reltool-1.0 runtime_tools-2.0 sasl-4.2.1 ssh-5.0.1 syntax_tools-3.1 tftp-1.1 tools-3.6 wx-2.3 xmerl-1.3.32 :
     OTP-26.0.2 : compiler-8.3.2 erts-14.0.2 kernel-9.0.2 ssh-5.0.1 ssl-11.0.2 stdlib-5.0.2 # asn1-5.1 common_test-1.25 crypto-5.2 debugger-5.3.1 dialyzer-5.1 diameter-2.3 edoc-1.2 eldap-1.2.11 erl_docgen-1.5 erl_interface-5.4 et-1.7 eunit-2.8.2 ftp-1.2 inets-9.0.1 jinterface-1.14 megaco-4.4.4 mnesia-4.22 observer-2.15 odbc-2.14.1 os_mon-2.9 parsetools-2.5 public_key-1.14 reltool-1.0 runtime_tools-2.0 sasl-4.2.1 snmp-5.14 syntax_tools-3.1 tftp-1.1 tools-3.6 wx-2.3 xmerl-1.3.32 :
     OTP-26.0.1 : compiler-8.3.1 erts-14.0.1 inets-9.0.1 kernel-9.0.1 ssl-11.0.1 stdlib-5.0.1 xmerl-1.3.32 # asn1-5.1 common_test-1.25 crypto-5.2 debugger-5.3.1 dialyzer-5.1 diameter-2.3 edoc-1.2 eldap-1.2.11 erl_docgen-1.5 erl_interface-5.4 et-1.7 eunit-2.8.2 ftp-1.2 jinterface-1.14 megaco-4.4.4 mnesia-4.22 observer-2.15 odbc-2.14.1 os_mon-2.9 parsetools-2.5 public_key-1.14 reltool-1.0 runtime_tools-2.0 sasl-4.2.1 snmp-5.14 ssh-5.0 syntax_tools-3.1 tftp-1.1 tools-3.6 wx-2.3 :
     OTP-26.0 : asn1-5.1 common_test-1.25 compiler-8.3 crypto-5.2 dialyzer-5.1 diameter-2.3 erl_docgen-1.5 erl_interface-5.4 erts-14.0 et-1.7 ftp-1.2 inets-9.0 jinterface-1.14 kernel-9.0 megaco-4.4.4 mnesia-4.22 observer-2.15 odbc-2.14.1 os_mon-2.9 parsetools-2.5 public_key-1.14 reltool-1.0 runtime_tools-2.0 sasl-4.2.1 snmp-5.14 ssh-5.0 ssl-11.0 stdlib-5.0 syntax_tools-3.1 tftp-1.1 tools-3.6 wx-2.3 # debugger-5.3.1 edoc-1.2 eldap-1.2.11 eunit-2.8.2 xmerl-1.3.31 :
     OTP-23.3.4.20 : ssh-4.11.1.7 # asn1-5.0.15.1 common_test-1.20.2.3 compiler-7.6.9.3 crypto-4.9.0.4 debugger-5.0 dialyzer-4.3.1.2 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 erts-11.2.2.18 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.3 jinterface-1.11.1.1 kernel-7.3.1.7 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssl-10.3.1.5 stdlib-3.14.2.3 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27.1 :
     OTP-23.3.4.19 : compiler-7.6.9.3 erts-11.2.2.18 stdlib-3.14.2.3 xmerl-1.3.27.1 # asn1-5.0.15.1 common_test-1.20.2.3 crypto-4.9.0.4 debugger-5.0 dialyzer-4.3.1.2 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.3 jinterface-1.11.1.1 kernel-7.3.1.7 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.6 ssl-10.3.1.5 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 :
     OTP-23.3.4.18 : dialyzer-4.3.1.2 erts-11.2.2.17 kernel-7.3.1.7 # asn1-5.0.15.1 common_test-1.20.2.3 compiler-7.6.9.2 crypto-4.9.0.4 debugger-5.0 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.3 jinterface-1.11.1.1 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.6 ssl-10.3.1.5 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.17 : erts-11.2.2.16 inets-7.3.2.3 kernel-7.3.1.6 ssl-10.3.1.5 # asn1-5.0.15.1 common_test-1.20.2.3 compiler-7.6.9.2 crypto-4.9.0.4 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 jinterface-1.11.1.1 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.6 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.16 : crypto-4.9.0.4 erts-11.2.2.15 ssl-10.3.1.4 # asn1-5.0.15.1 common_test-1.20.2.3 compiler-7.6.9.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1.1 kernel-7.3.1.5 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.6 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.15 : crypto-4.9.0.3 erts-11.2.2.14 ssh-4.11.1.6 ssl-10.3.1.3 # asn1-5.0.15.1 common_test-1.20.2.3 compiler-7.6.9.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1.1 kernel-7.3.1.5 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.14 : compiler-7.6.9.2 erts-11.2.2.13 # asn1-5.0.15.1 common_test-1.20.2.3 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1.1 kernel-7.3.1.5 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.5 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.13 : erts-11.2.2.12 # asn1-5.0.15.1 common_test-1.20.2.3 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1.1 kernel-7.3.1.5 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.5 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.12 : common_test-1.20.2.3 erts-11.2.2.11 jinterface-1.11.1.1 kernel-7.3.1.5 # asn1-5.0.15.1 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.5 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.11 : erts-11.2.2.10 ssh-4.11.1.5 # asn1-5.0.15.1 common_test-1.20.2.2 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1 kernel-7.3.1.4 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.10 : erts-11.2.2.9 # asn1-5.0.15.1 common_test-1.20.2.2 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1 kernel-7.3.1.4 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.4 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.9 : erts-11.2.2.8 # asn1-5.0.15.1 common_test-1.20.2.2 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1 kernel-7.3.1.4 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.4 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.8 : erts-11.2.2.7 ssh-4.11.1.4 # asn1-5.0.15.1 common_test-1.20.2.2 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.2 jinterface-1.11.1 kernel-7.3.1.4 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.7 : erts-11.2.2.6 inets-7.3.2.2 kernel-7.3.1.4 # asn1-5.0.15.1 common_test-1.20.2.2 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 jinterface-1.11.1 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.3 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.6 : erts-11.2.2.5 kernel-7.3.1.3 # asn1-5.0.15.1 common_test-1.20.2.2 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.1 jinterface-1.11.1 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10.0.1 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.3 ssl-10.3.1.2 stdlib-3.14.2.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.5 : asn1-5.0.15.1 common_test-1.20.2.2 erts-11.2.2.4 public_key-1.10.0.1 ssl-10.3.1.2 stdlib-3.14.2.2 # compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2.1 jinterface-1.11.1 kernel-7.3.1.2 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.3 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.4 : dialyzer-4.3.1.1 inets-7.3.2.1 # asn1-5.0.15 common_test-1.20.2.1 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 erts-11.2.2.3 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 jinterface-1.11.1 kernel-7.3.1.2 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssh-4.11.1.3 ssl-10.3.1.1 stdlib-3.14.2.1 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.3 : erts-11.2.2.3 kernel-7.3.1.2 ssh-4.11.1.3 # asn1-5.0.15 common_test-1.20.2.1 compiler-7.6.9.1 crypto-4.9.0.2 debugger-5.0 dialyzer-4.3.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11.1 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 ssl-10.3.1.1 stdlib-3.14.2.1 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.2 : compiler-7.6.9.1 crypto-4.9.0.2 erts-11.2.2.2 kernel-7.3.1.1 ssh-4.11.1.2 ssl-10.3.1.1 stdlib-3.14.2.1 # asn1-5.0.15 common_test-1.20.2.1 debugger-5.0 dialyzer-4.3.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11.1 megaco-3.19.5.1 mnesia-4.19 observer-2.9.5 odbc-2.13.3.1 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8.0.1 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3.1 xmerl-1.3.27 :
     OTP-23.3.4.1 : common_test-1.20.2.1 crypto-4.9.0.1 erl_interface-4.0.3.1 erts-11.2.2.1 megaco-3.19.5.1 odbc-2.13.3.1 snmp-5.8.0.1 ssh-4.11.1.1 wx-1.9.3.1 # asn1-5.0.15 compiler-7.6.9 debugger-5.0 dialyzer-4.3.1 diameter-2.2.4 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11.1 kernel-7.3.1 mnesia-4.19 observer-2.9.5 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 ssl-10.3.1 stdlib-3.14.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 xmerl-1.3.27 :
     OTP-23.3.4 : compiler-7.6.9 diameter-2.2.4 erts-11.2.2 # asn1-5.0.15 common_test-1.20.2 crypto-4.9 debugger-5.0 dialyzer-4.3.1 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.3 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11.1 kernel-7.3.1 megaco-3.19.5 mnesia-4.19 observer-2.9.5 odbc-2.13.3 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 runtime_tools-1.16.1 sasl-4.0.2 snmp-5.8 ssh-4.11.1 ssl-10.3.1 stdlib-3.14.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3 xmerl-1.3.27 :
     OTP-23.3.3 : common_test-1.20.2 compiler-7.6.8 erl_interface-4.0.3 kernel-7.3.1 runtime_tools-1.16.1 # asn1-5.0.15 crypto-4.9 debugger-5.0 dialyzer-4.3.1 diameter-2.2.3 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erts-11.2.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11.1 megaco-3.19.5 mnesia-4.19 observer-2.9.5 odbc-2.13.3 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 sasl-4.0.2 snmp-5.8 ssh-4.11.1 ssl-10.3.1 stdlib-3.14.2 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3 xmerl-1.3.27 :
     OTP-23.3.2 : asn1-5.0.15 common_test-1.20.1 erts-11.2.1 ssl-10.3.1 stdlib-3.14.2 xmerl-1.3.27 # compiler-7.6.7 crypto-4.9 debugger-5.0 dialyzer-4.3.1 diameter-2.2.3 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.2 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11.1 kernel-7.3 megaco-3.19.5 mnesia-4.19 observer-2.9.5 odbc-2.13.3 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 runtime_tools-1.16 sasl-4.0.2 snmp-5.8 ssh-4.11.1 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3 :
     OTP-23.3.1 : ssh-4.11.1 # asn1-5.0.14 common_test-1.20 compiler-7.6.7 crypto-4.9 debugger-5.0 dialyzer-4.3.1 diameter-2.2.3 edoc-0.12 eldap-1.2.9 erl_docgen-1.0.2 erl_interface-4.0.2 erts-11.2 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11.1 kernel-7.3 megaco-3.19.5 mnesia-4.19 observer-2.9.5 odbc-2.13.3 os_mon-2.6.1 parsetools-2.2 public_key-1.10 reltool-0.8 runtime_tools-1.16 sasl-4.0.2 snmp-5.8 ssl-10.3 stdlib-3.14.1 syntax_tools-2.5 tftp-1.0.2 tools-3.4.4 wx-1.9.3 xmerl-1.3.26 :
     OTP-23.3 : common_test-1.20 compiler-7.6.7 crypto-4.9 dialyzer-4.3.1 eldap-1.2.9 erts-11.2 jinterface-1.11.1 kernel-7.3 mnesia-4.19 odbc-2.13.3 public_key-1.10 runtime_tools-1.16 sasl-4.0.2 snmp-5.8 ssh-4.11 ssl-10.3 stdlib-3.14.1 syntax_tools-2.5 tools-3.4.4 wx-1.9.3 # asn1-5.0.14 debugger-5.0 diameter-2.2.3 edoc-0.12 erl_docgen-1.0.2 erl_interface-4.0.2 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 megaco-3.19.5 observer-2.9.5 os_mon-2.6.1 parsetools-2.2 reltool-0.8 tftp-1.0.2 xmerl-1.3.26 :
     OTP-23.2.7.5 : ssl-10.2.4.4 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2.1 erts-11.1.8 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11 kernel-7.2.1 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssh-4.10.8 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.3 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.7.4 : ssl-10.2.4.3 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2.1 erts-11.1.8 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11 kernel-7.2.1 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssh-4.10.8 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.3 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.7.3 : erl_interface-4.0.2.1 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erts-11.1.8 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11 kernel-7.2.1 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssh-4.10.8 ssl-10.2.4.2 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.3 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.7.2 : ssl-10.2.4.2 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2 erts-11.1.8 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11 kernel-7.2.1 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssh-4.10.8 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.3 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.7.1 : ssl-10.2.4.1 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2 erts-11.1.8 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11 kernel-7.2.1 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssh-4.10.8 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.3 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.7 : kernel-7.2.1 ssl-10.2.4 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2 erts-11.1.8 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.2 jinterface-1.11 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssh-4.10.8 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.3 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.6 : inets-7.3.2 ssh-4.10.8 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2 erts-11.1.8 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 jinterface-1.11 kernel-7.2 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssl-10.2.3 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.3 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.5 : erts-11.1.8 ssl-10.2.3 tools-3.4.3 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.1 jinterface-1.11 kernel-7.2 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7.3 ssh-4.10.7 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.4 : snmp-5.7.3 ssl-10.2.2 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.3 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2 erts-11.1.7 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.1 jinterface-1.11 kernel-7.2 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 ssh-4.10.7 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.2 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.3 : crypto-4.8.3 erts-11.1.7 snmp-5.7.2 ssh-4.10.7 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.2 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.1 jinterface-1.11 kernel-7.2 megaco-3.19.5 mnesia-4.18.1 observer-2.9.5 odbc-2.13.2 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 ssl-10.2.1 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.2 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.2 : crypto-4.8.2 erl_interface-4.0.2 erts-11.1.6 megaco-3.19.5 odbc-2.13.2 snmp-5.7.1 ssl-10.2.1 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.1 jinterface-1.11 kernel-7.2 mnesia-4.18.1 observer-2.9.5 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 ssh-4.10.6 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.2 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2.1 : erts-11.1.5 # asn1-5.0.14 common_test-1.19.1 compiler-7.6.6 crypto-4.8.1 debugger-5.0 dialyzer-4.3 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.2 erl_interface-4.0.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3.1 jinterface-1.11 kernel-7.2 megaco-3.19.4 mnesia-4.18.1 observer-2.9.5 odbc-2.13.1 os_mon-2.6.1 parsetools-2.2 public_key-1.9.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.7 ssh-4.10.6 ssl-10.2 stdlib-3.14 syntax_tools-2.4 tftp-1.0.2 tools-3.4.2 wx-1.9.2 xmerl-1.3.26 :
     OTP-23.2 : common_test-1.19.1 compiler-7.6.6 crypto-4.8.1 dialyzer-4.3 erl_docgen-1.0.2 erts-11.1.4 inets-7.3.1 kernel-7.2 megaco-3.19.4 mnesia-4.18.1 public_key-1.9.2 snmp-5.7 ssh-4.10.6 ssl-10.2 stdlib-3.14 syntax_tools-2.4 tools-3.4.2 wx-1.9.2 xmerl-1.3.26 # asn1-5.0.14 debugger-5.0 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_interface-4.0.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 jinterface-1.11 observer-2.9.5 odbc-2.13.1 os_mon-2.6.1 parsetools-2.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 tftp-1.0.2 :
     OTP-23.1.5 : ssh-4.10.5 # asn1-5.0.14 common_test-1.19 compiler-7.6.5 crypto-4.8 debugger-5.0 dialyzer-4.2.1 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.1 erl_interface-4.0.1 erts-11.1.3 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3 jinterface-1.11 kernel-7.1 megaco-3.19.3 mnesia-4.18 observer-2.9.5 odbc-2.13.1 os_mon-2.6.1 parsetools-2.2 public_key-1.9.1 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.6.1 ssl-10.1 stdlib-3.13.2 syntax_tools-2.3.1 tftp-1.0.2 tools-3.4.1 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.1.4.1 : ssh-4.10.4.1 # asn1-5.0.14 common_test-1.19 compiler-7.6.5 crypto-4.8 debugger-5.0 dialyzer-4.2.1 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.1 erl_interface-4.0.1 erts-11.1.3 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3 jinterface-1.11 kernel-7.1 megaco-3.19.3 mnesia-4.18 observer-2.9.5 odbc-2.13.1 os_mon-2.6.1 parsetools-2.2 public_key-1.9.1 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.6.1 ssl-10.1 stdlib-3.13.2 syntax_tools-2.3.1 tftp-1.0.2 tools-3.4.1 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.1.4 : ssh-4.10.4 # asn1-5.0.14 common_test-1.19 compiler-7.6.5 crypto-4.8 debugger-5.0 dialyzer-4.2.1 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.1 erl_interface-4.0.1 erts-11.1.3 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3 jinterface-1.11 kernel-7.1 megaco-3.19.3 mnesia-4.18 observer-2.9.5 odbc-2.13.1 os_mon-2.6.1 parsetools-2.2 public_key-1.9.1 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.6.1 ssl-10.1 stdlib-3.13.2 syntax_tools-2.3.1 tftp-1.0.2 tools-3.4.1 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.1.3 : erts-11.1.3 ssh-4.10.3 # asn1-5.0.14 common_test-1.19 compiler-7.6.5 crypto-4.8 debugger-5.0 dialyzer-4.2.1 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.1 erl_interface-4.0.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3 jinterface-1.11 kernel-7.1 megaco-3.19.3 mnesia-4.18 observer-2.9.5 odbc-2.13.1 os_mon-2.6.1 parsetools-2.2 public_key-1.9.1 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.6.1 ssl-10.1 stdlib-3.13.2 syntax_tools-2.3.1 tftp-1.0.2 tools-3.4.1 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.1.2 : compiler-7.6.5 erts-11.1.2 # asn1-5.0.14 common_test-1.19 crypto-4.8 debugger-5.0 dialyzer-4.2.1 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.1 erl_interface-4.0.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3 jinterface-1.11 kernel-7.1 megaco-3.19.3 mnesia-4.18 observer-2.9.5 odbc-2.13.1 os_mon-2.6.1 parsetools-2.2 public_key-1.9.1 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.6.1 ssh-4.10.2 ssl-10.1 stdlib-3.13.2 syntax_tools-2.3.1 tftp-1.0.2 tools-3.4.1 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.1.1 : compiler-7.6.4 erts-11.1.1 os_mon-2.6.1 public_key-1.9.1 ssh-4.10.2 # asn1-5.0.14 common_test-1.19 crypto-4.8 debugger-5.0 dialyzer-4.2.1 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0.1 erl_interface-4.0.1 et-1.6.4 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3 jinterface-1.11 kernel-7.1 megaco-3.19.3 mnesia-4.18 observer-2.9.5 odbc-2.13.1 parsetools-2.2 reltool-0.8 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.6.1 ssl-10.1 stdlib-3.13.2 syntax_tools-2.3.1 tftp-1.0.2 tools-3.4.1 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.1 : asn1-5.0.14 compiler-7.6.3 crypto-4.8 dialyzer-4.2.1 erl_docgen-1.0.1 erl_interface-4.0.1 erts-11.1 eunit-2.6 ftp-1.0.5 hipe-4.0.1 inets-7.3 kernel-7.1 megaco-3.19.3 mnesia-4.18 observer-2.9.5 odbc-2.13.1 os_mon-2.6 public_key-1.9 runtime_tools-1.15.1 sasl-4.0.1 snmp-5.6.1 ssh-4.10.1 ssl-10.1 stdlib-3.13.2 syntax_tools-2.3.1 tools-3.4.1 # common_test-1.19 debugger-5.0 diameter-2.2.3 edoc-0.12 eldap-1.2.8 et-1.6.4 jinterface-1.11 parsetools-2.2 reltool-0.8 tftp-1.0.2 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.0.4 : erts-11.0.4 megaco-3.19.2 stdlib-3.13.1 # asn1-5.0.13 common_test-1.19 compiler-7.6.2 crypto-4.7 debugger-5.0 dialyzer-4.2 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0 erl_interface-4.0 et-1.6.4 eunit-2.5 ftp-1.0.4 hipe-4.0 inets-7.2 jinterface-1.11 kernel-7.0 mnesia-4.17 observer-2.9.4 odbc-2.13 os_mon-2.5.2 parsetools-2.2 public_key-1.8 reltool-0.8 runtime_tools-1.15 sasl-4.0 snmp-5.6 ssh-4.10 ssl-10.0 syntax_tools-2.3 tftp-1.0.2 tools-3.4 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.0.3 : compiler-7.6.2 erts-11.0.3 # asn1-5.0.13 common_test-1.19 crypto-4.7 debugger-5.0 dialyzer-4.2 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0 erl_interface-4.0 et-1.6.4 eunit-2.5 ftp-1.0.4 hipe-4.0 inets-7.2 jinterface-1.11 kernel-7.0 megaco-3.19.1 mnesia-4.17 observer-2.9.4 odbc-2.13 os_mon-2.5.2 parsetools-2.2 public_key-1.8 reltool-0.8 runtime_tools-1.15 sasl-4.0 snmp-5.6 ssh-4.10 ssl-10.0 stdlib-3.13 syntax_tools-2.3 tftp-1.0.2 tools-3.4 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.0.2 : erts-11.0.2 megaco-3.19.1 # asn1-5.0.13 common_test-1.19 compiler-7.6.1 crypto-4.7 debugger-5.0 dialyzer-4.2 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0 erl_interface-4.0 et-1.6.4 eunit-2.5 ftp-1.0.4 hipe-4.0 inets-7.2 jinterface-1.11 kernel-7.0 mnesia-4.17 observer-2.9.4 odbc-2.13 os_mon-2.5.2 parsetools-2.2 public_key-1.8 reltool-0.8 runtime_tools-1.15 sasl-4.0 snmp-5.6 ssh-4.10 ssl-10.0 stdlib-3.13 syntax_tools-2.3 tftp-1.0.2 tools-3.4 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.0.1 : compiler-7.6.1 erts-11.0.1 # asn1-5.0.13 common_test-1.19 crypto-4.7 debugger-5.0 dialyzer-4.2 diameter-2.2.3 edoc-0.12 eldap-1.2.8 erl_docgen-1.0 erl_interface-4.0 et-1.6.4 eunit-2.5 ftp-1.0.4 hipe-4.0 inets-7.2 jinterface-1.11 kernel-7.0 megaco-3.19 mnesia-4.17 observer-2.9.4 odbc-2.13 os_mon-2.5.2 parsetools-2.2 public_key-1.8 reltool-0.8 runtime_tools-1.15 sasl-4.0 snmp-5.6 ssh-4.10 ssl-10.0 stdlib-3.13 syntax_tools-2.3 tftp-1.0.2 tools-3.4 wx-1.9.1 xmerl-1.3.25 :
     OTP-23.0 : asn1-5.0.13 common_test-1.19 compiler-7.6 crypto-4.7 debugger-5.0 dialyzer-4.2 edoc-0.12 erl_docgen-1.0 erl_interface-4.0 erts-11.0 eunit-2.5 hipe-4.0 inets-7.2 jinterface-1.11 kernel-7.0 megaco-3.19 mnesia-4.17 observer-2.9.4 odbc-2.13 os_mon-2.5.2 parsetools-2.2 public_key-1.8 runtime_tools-1.15 sasl-4.0 snmp-5.6 ssh-4.10 ssl-10.0 stdlib-3.13 syntax_tools-2.3 tools-3.4 wx-1.9.1 xmerl-1.3.25 # diameter-2.2.3 eldap-1.2.8 et-1.6.4 ftp-1.0.4 reltool-0.8 tftp-1.0.2 :
     """,
    _ = cmd("mv otp_versions.table otp_versions.table.backup"),
    Module = ?MODULE,
    file:write_file("otp_versions.table", OTPTable),
    Result = apply(Module, F, []),
    cmd("mv otp_versions.table.backup otp_versions.table"),
    Result.
